<?php

namespace Drupal\static_facet_values\Plugin\facets\processor;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\facets\FacetInterface;
use Drupal\facets\Processor\BuildProcessorInterface;
use Drupal\facets\Processor\ProcessorPluginBase;
use Drupal\static_facet_values\StaticFacetValuesCollectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a processor that sets static values to a facet.
 *
 * The facet results are provided by a service implementing
 * \Drupal\static_facet_values\StaticFacetValuesServiceInterface.
 *
 * @FacetsProcessor(
 *   id = "sfv_static_values",
 *   label = @Translation("Static facet values"),
 *   description = @Translation("Add static values to the facet widget."),
 *   stages = {
 *     "build" = 50
 *   }
 * )
 */
class StaticValuesProcessor extends ProcessorPluginBase implements BuildProcessorInterface, ContainerFactoryPluginInterface {

  const OPTION_NONE = '_none';

  /**
   * Constructs a StaticValuesProcessor object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\static_facet_values\StaticFacetValuesCollectionInterface $serviceCollection
   *   The static facet values service collection.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    $plugin_definition,
    protected StaticFacetValuesCollectionInterface $serviceCollection,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    // @phpstan-ignore-next-line
    $self = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('static_facet_values.collection')
    );

    return $self;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'static_value_service' => self::OPTION_NONE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, FacetInterface $facet): array {
    $build = [];
    $config = $this->getConfiguration();

    $services_options = [
      self::OPTION_NONE => $this->t('- Select -'),
    ] + $this->serviceCollection->listServicesAsOptions();

    $static_value_service_id = $config['static_value_service'] ?? '';
    $build['static_value_service'] = [
      '#title' => $this->t('Static value service'),
      '#type' => 'select',
      '#options' => $services_options,
      '#default_value' => $static_value_service_id,
      '#description' => $this->t('Enter the ID of the static value service.'),
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function build(FacetInterface $facet, array $results): array {
    $config = $this->getConfiguration();
    // @phpstan-ignore-next-line
    $container = \Drupal::getContainer();
    $static_value_service_id = $config['static_value_service'] ?? '';
    if ($static_value_service_id === self::OPTION_NONE) {
      return $results;
    }

    if (!$container->has($static_value_service_id)) {
      return $results;
    }

    /** @var \Drupal\static_facet_values\StaticFacetValuesServiceInterface $service */
    $service = $container->get($static_value_service_id);

    return $service->getResults($facet, $results);
  }

}
