<?php

namespace Drupal\Tests\static_facet_values\Unit\Plugin\processor;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\facets\Entity\Facet;
use Drupal\facets\Result\Result;
use Drupal\sfv_processor_collection\Facets\StaticDays;
use Drupal\sfv_processor_collection\Facets\StaticNumbers;
use Drupal\static_facet_values\StaticFacetValuesServiceInterface;
use Drupal\static_facet_values\Plugin\facets\processor\StaticValuesProcessor;
use Drupal\static_facet_values\StaticFacetValuesCollectionInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\Tests\static_facet_values\Unit\StaticFacetValuesTestInterface;

/**
 * Unit test for the StaticValuesProcessor.
 *
 * @coversDefaultClass \Drupal\static_facet_values\Plugin\facets\processor\StaticValuesProcessor
 * @group static_facet_values
 */
class StaticValuesProcessorTest extends UnitTestCase implements StaticFacetValuesTestInterface {

  /**
   * The mocked service container.
   *
   * @var \Drupal\Core\DependencyInjection\ContainerBuilder|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $container;

  /**
   * The mocked service collection.
   *
   * @var \Drupal\static_facet_values\StaticFacetValuesCollectionInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $serviceCollection;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->serviceCollection = $this->createMock(StaticFacetValuesCollectionInterface::class);

    // Create a mock container and set it as the main Drupal container. This
    // allows us to control which services are available during the test.
    $this->container = $this->createMock(ContainerBuilder::class);
    \Drupal::setContainer($this->container);
  }

  /**
   * Tests the build method with different static value services.
   *
   * @param array<string, string> $configuration
   *   The processor configuration.
   * @param string $serviceId
   *   The ID of the service to be retrieved from the container.
   * @param \Drupal\static_facet_values\StaticFacetValuesServiceInterface $serviceObject
   *   The service object that the container should return.
   * @param \Drupal\facets\Result\Result[] $inputResults
   *   The initial array of results to pass to the processor.
   * @param array<int, \Drupal\facets\Result\Result> $expectedResults
   *   The expected array of results after the processor runs.
   * @param string $message
   *   The message to display on failure.
   *
   * @dataProvider processorConfigurationProvider
   */
  public function testBuild(array $configuration, string $serviceId, StaticFacetValuesServiceInterface $serviceObject, array $inputResults, array $expectedResults, string $message): void {
    // Configure the mock container to recognize and return the service.
    $this->container->method('has')->with($serviceId)->willReturn(TRUE);
    $this->container->method('get')->with($serviceId)->willReturn($serviceObject);

    // The processor's build() method requires a Facet object.
    $facet = new Facet([], 'facets_facet');

    // Instantiate the processor with the test-specific configuration.
    // @phpstan-ignore-next-line
    $processor = new StaticValuesProcessor($configuration, 'sfv_static_values', [], $this->serviceCollection);
    $actualResults = $processor->build($facet, $inputResults);

    $this->assertEquals($expectedResults, $actualResults, $message);
  }

  /**
   * Provides test cases for different processor configurations.
   *
   * @return iterable<string, array>
   *   A list of test cases.
   */
  public static function processorConfigurationProvider(): iterable {
    $facet = new Facet([], 'facets_facet');

    // Case 1: Test with the "numbers" service.
    $numbers_service_id = self::VALUES_SERVICE_NUMBERS;
    yield 'Using StaticNumbers' => [
      'configuration' => ['static_value_service' => $numbers_service_id],
      'serviceId' => $numbers_service_id,
      'serviceObject' => new StaticNumbers(),
      'inputResults' => [
        new Result($facet, 'a', 'a', 1),
        new Result($facet, 'b', 'b', 2),
        new Result($facet, 'c', 'c', 3),
      ],
      'expectedResults' => [
        1 => new Result($facet, 1, 'one', 1),
        2 => new Result($facet, 2, 'two', 1),
        3 => new Result($facet, 3, 'three', 1),
      ],
      'message' => 'Processor failed to generate correct results using StaticNumbers.',
    ];

    // Case 2: Test with the "days" service.
    $daysServiceId = self::VALUES_SERVICE_DAYS;
    yield 'Using StaticDays' => [
      'configuration' => ['static_value_service' => $daysServiceId],
      'serviceId' => $daysServiceId,
      'serviceObject' => new StaticDays(),
      'inputResults' => [
        new Result($facet, 'a', 'a', 11),
        new Result($facet, 'b', 'b', 22),
        new Result($facet, 'c', 'c', 33),
        new Result($facet, 'd', 'd', 44),
      ],
      'expectedResults' => [
        0 => new Result($facet, 0, 'Monday', 1),
        1 => new Result($facet, 1, 'Tuesday', 1),
        2 => new Result($facet, 2, 'Wednesday', 1),
        3 => new Result($facet, 3, 'Thursday', 1),
      ],
      'message' => 'Processor failed to generate correct results using StaticDays.',
    ];
  }

}
