<?php

namespace Drupal\staticjson\EventSubscriber;

use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Entity\EntityInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\path_alias\Entity\PathAlias;
/**
 * staticjson event subscriber.
 */
class StaticjsonSubscriber implements EventSubscriberInterface {

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

    /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  public function __construct(ConfigFactoryInterface $config_factory,MessengerInterface $messenger) {
    $this->configFactory = $config_factory; 
    $this->messenger = $messenger;
  }

  /**
   * Kernel request event handler.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   Response event.
   */
  public function onKernelRequest(RequestEvent $event) {
    $request = $event->getRequest();

    $config = $this->configFactory->get('staticjson.settings');
    $selected_content_types = array_filter($config->get('content_types') ?? []);
    $method = $request->getMethod();
    $is_read = in_array($method, ['GET', 'HEAD']);
    $is_write = in_array($method, ['POST', 'PUT', 'PATCH', 'DELETE']);


    $detected = $this->detectEntityFromRoute();
    if (empty($detected['entity_type']) && empty($detected['bundle'])){ //&& $is_write){
      return;
    }

    static $lookup = NULL;
    if ($lookup === NULL) {
      $lookup = array_flip($selected_content_types);
    }

    // ❗ Validar si el bundle está dentro de la selección
    if (isset($lookup[$detected['bundle']])) {
      $entity = $detected['entity'];
      //$json_data = $this->entityToJsonDeep($entity,);
      //$this->saveEntityJson($entity,$json_data);
    }


    //$this->messenger->addStatus(__FUNCTION__);
  }

   /**
   * Convierte una entidad a JSON cargando referencias profundamente.
   */


  protected function entityToJsonDeep(EntityInterface $entity, int $depth = 0, array &$visited = []) {

    // -------------------------------
    // CONFIGURACIÓN SEGURA
    // -------------------------------
    $max_depth = 2; // limite QUEMADO
    $exclude_types = ['user', 'file']; // tipos excluidos
    $sensitive_fields = [
      'pass', 'password', 'mail', 'email', 'user_password',
      'authmap', 'roles', 'permissions', 'token', 'secret',
      'private_key', 'api_key', 'consumer_key', 'consumer_secret',
    ];

    $entity_type = $entity->getEntityTypeId();
    $entity_id   = $entity->id();

    // -------------------------------
    // EXCLUSIONES DE ENTIDADES COMPLETAS
    // -------------------------------
    if (in_array($entity_type, $exclude_types, TRUE)) {
      return [
        '_entity_type' => $entity_type,
        '_id' => $entity_id,
        '_excluded' => TRUE,
      ];
    }

    // -------------------------------
    // ANTI-RECURSIÓN (ciclos)
    // -------------------------------
    $key = "$entity_type:$entity_id";

    if (isset($visited[$key])) {
      return [
        '_entity_type' => $entity_type,
        '_id' => $entity_id,
        '_recursive_reference' => TRUE,
      ];
    }

    $visited[$key] = TRUE;

    // -------------------------------
    // LÍMITE DE PROFUNDIDAD
    // -------------------------------
    if ($depth >= $max_depth) {
      return [
        '_entity_type' => $entity_type,
        '_id' => $entity_id,
        '_depth_limit' => TRUE,
      ];
    }

    // -------------------------------
    // BASE JSON
    // -------------------------------
    $output = [
      '_entity_type' => $entity_type,
      '_bundle'      => method_exists($entity, 'bundle') ? $entity->bundle() : NULL,
      '_id'          => $entity_id,
    ];

    // -------------------------------
    // SI NO ES FIELDABLE → metadata
    // -------------------------------
    if (!$entity instanceof FieldableEntityInterface) {
      $output['_raw'] = $entity->toArray();
      return $output;
    }

    // -------------------------------
    // PROCESAR CAMPOS
    // -------------------------------
    foreach ($entity->getFields() as $field_name => $field) {

      // 1) Proteger campos sensibles
      if (in_array($field_name, $sensitive_fields, TRUE)) {
        $output[$field_name] = '[PROTEGIDO]';
        continue;
      }

      $definition = $field->getFieldDefinition();
      $type = $definition->getType();

      // --------------------------------
      // Caso 1: campo NO referencia
      // --------------------------------
      if ($type !== 'entity_reference' && $type !== 'entity_reference_revisions') {
        $output[$field_name] = $field->getValue();
        continue;
      }

      // --------------------------------
      // Caso 2: campo de referencia
      // --------------------------------
      $target_type = $definition->getSetting('target_type');

      if (!$target_type) {
        $output[$field_name] = $field->getValue();
        continue;
      }

      $items = [];
      foreach ($field->getValue() as $item) {

        if (!isset($item['target_id'])) {
          $items[] = $item;
          continue;
        }

        $target = \Drupal::entityTypeManager()
          ->getStorage($target_type)
          ->load($item['target_id']);

        if ($target instanceof EntityInterface) {
          // Recursión segura
          $items[] = $this->entityToJsonDeep($target, $depth + 1, $visited);
        }
        else {
          $items[] = $item;
        }
      }

      $output[$field_name] = $items;
    }

    return $output;
  }


   /**
   * Save json to file.
   */
  protected function entityToJsonSafe(EntityInterface $entity, $depth = 0, $max_depth = 2, &$seen = []) {

    if ($depth > $max_depth) {
      return "[MAX_DEPTH_REACHED]";
    }

    $uuid = $entity->uuid();
    if (isset($seen[$uuid])) {
      return "[RECURSION_BLOCKED]";
    }
    $seen[$uuid] = TRUE;

    // Campos sensibles a excluir
    $excluded_fields = [
      'pass', 'password', 'mail',
      'revision_log', 'revision_user',
    ];

    $data = [
      '_entity_type' => $entity->getEntityTypeId(),
      '_bundle'      => $entity->bundle(),
      '_id'          => $entity->id(),
      '_uuid'        => $uuid,
    ];

    foreach ($entity->getFields() as $field_name => $field) {

      if (in_array($field_name, $excluded_fields)) {
        continue;
      }

      // Si el campo es vacío
      if ($field->isEmpty()) {
        $data[$field_name] = NULL;
        continue;
      }

      // Si el campo referencia otra entidad
      if ($field->getFieldDefinition()->getType() === 'entity_reference') {
        $items = [];
        foreach ($field->referencedEntities() as $ref) {
          $items[] = $this->entityToJsonSafe($ref, $depth + 1, $max_depth, $seen);
        }
        $data[$field_name] = $items;
        continue;
      }

      // Campo normal
      $data[$field_name] = $field->getValue();
    }

    return $data;
  }

protected function saveEntityJson(EntityInterface $entity, array $json_data) {
  $bundle = $entity->bundle();
  $id = $entity->id();

  // Convertir a JSON
  $json_string = json_encode($json_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

  // -----------------------------
  // Ruta del módulo
  // -----------------------------
  $module_path = \Drupal::service('extension.list.module')
    ->getPath('staticjson');

  // Base /json/
  $base_dir = DRUPAL_ROOT . '/' . $module_path . '/json/';

  // Carpeta por bundle SIEMPRE termina en "/"
  $target_dir = $base_dir . $bundle . '/';

  // Crear carpeta si no existe
  \Drupal::service('file_system')->prepareDirectory(
    $target_dir,
    FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
  );

  // -----------------------------
  // Guardar archivo principal por ID
  // -----------------------------
  $id_filepath = $target_dir .'/'. $id . '.json';
  file_put_contents($id_filepath, $json_string);

  // -----------------------------
  // Alias
  // -----------------------------
  $alias = NULL;

  if ($entity->hasLinkTemplate('canonical')) {
    $internal = $entity->toUrl('canonical')->getInternalPath();
    $alias_raw = \Drupal::service('path_alias.manager')->getAliasByPath('/' . $internal);

    if (!empty($alias_raw) && $alias_raw !== '/' . $internal) {
      $alias = ltrim($alias_raw, '/');
      $alias = str_replace('/', '_', $alias); // limpiar alias
    }
  }

  if ($alias) {
    $alias_filepath = $target_dir .'/'. $alias . '.json';
    file_put_contents($alias_filepath, $json_string);
  }
}





  /**
 * Detecta la entidad actual según la ruta.
 *
 * @return array
 *   [
 *     'entity_type' => string|null,
 *     'bundle' => string|null,
 *     'id' => int|string|null,
 *     'entity' => object|null,
 *   ]
 */
  protected function detectEntityFromRoute() {
    $route_match = \Drupal::routeMatch();

    // Template de salida.
    $result = [
      'entity_type' => null,
      'bundle' => null,
      'id' => null,
      'entity' => null,
    ];

    // -------------------------------
    // NODES
    // -------------------------------
    $node = $route_match->getParameter('node');
    if ($node instanceof \Drupal\node\NodeInterface) {
      $result['entity_type'] = 'node';
      $result['bundle'] = $node->bundle();
      $result['id'] = $node->id();
      $result['entity'] = $node;
      return $result;
    }

    // -------------------------------
    // TAXONOMY TERMS
    // -------------------------------
    $term = $route_match->getParameter('taxonomy_term');
    if ($term instanceof \Drupal\taxonomy\TermInterface) {
      $result['entity_type'] = 'taxonomy_term';
      $result['bundle'] = $term->bundle();
      $result['id'] = $term->id();
      $result['entity'] = $term;
      return $result;
    }

    // -------------------------------
    // MENUS
    // -------------------------------
    $menu = $route_match->getParameter('menu');
    if ($menu instanceof \Drupal\system\MenuInterface) {
      $result['entity_type'] = 'menu';
      $result['bundle'] = $menu->id();
      $result['id'] = $menu->id();
      $result['entity'] = $menu;
      return $result;
    }

    // -------------------------------
    // MENU LINK CONTENT
    // -------------------------------
    $menu_link = $route_match->getParameter('menu_link_content');
    if ($menu_link instanceof \Drupal\menu_link_content\MenuLinkContentInterface) {
      $result['entity_type'] = 'menu_link_content';
      $result['bundle'] = $menu_link->bundle();
      $result['id'] = $menu_link->id();
      $result['entity'] = $menu_link;
      return $result;
    }

    return $result;
  }


  /**
   * Kernel response event handler.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent  $event
   *   Response event.
   */
  public function onKernelResponse(ResponseEvent $event) {
    //$this->messenger->addStatus(__FUNCTION__);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      //KernelEvents::REQUEST => ['onKernelRequest'],
      //KernelEvents::RESPONSE => ['onKernelResponse'],
    ];
  }

}
