<?php

namespace Drupal\staticjson\Services;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpFoundation\Response;

class JsonStorageReader {

  protected $config;
  protected $fileSystem;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    FileSystemInterface $file_system
  ) {
    $this->config = $config_factory->get('staticjson.settings');
    $this->fileSystem = $file_system;
  }

  /**
   * Lee JSON desde storage local o remoto.
   */
  public function loadJson(string $bundle, string $id_or_alias): ?array {

    $storage_type = $this->config->get('storage') ?? 'local';

    switch ($storage_type) {

      case 'local':
        return $this->loadLocal($bundle, $id_or_alias);

      case 's3':
        return $this->loadS3($bundle, $id_or_alias);

      default:
        \Drupal::logger('staticjson')->error("Storage no soportado: @s", ['@s' => $storage_type]);
        return NULL;
    }
  }

  /**
   * Carga JSON desde filesystem del módulo.
   */
  private function loadLocal(string $bundle, string $file): ?array {
    $module_path = \Drupal::service('extension.list.module')->getPath('staticjson');

    $path = DRUPAL_ROOT . "/$module_path/json/$bundle/$file.json";

    if (!file_exists($path)) {
      \Drupal::logger('staticjson')->warning('Archivo no existe: @p', ['@p' => $path]);
      return NULL;
    }

    $content = file_get_contents($path);
    return json_decode($content, TRUE);
  }

  /**
   * Cargar desde S3 (requiere AWS SDK).
   */
  private function loadS3(string $bundle, string $file): ?array {

    try {
      $s3 = new \Aws\S3\S3Client([
        'version' => 'latest',
        'region' => $this->config->get('aws_region'),
        'credentials' => [
          'key' => $this->config->get('aws_key'),
          'secret' => $this->config->get('aws_secret'),
        ],
      ]);

      $bucket = $this->config->get('s3_bucket');
      $key = "staticjson/$bundle/$file.json";

      $result = $s3->getObject([
        'Bucket' => $bucket,
        'Key' => $key,
      ]);

      return json_decode($result['Body'], TRUE);

    } catch (\Exception $e) {
      \Drupal::logger('staticjson')->error('Error S3: @e', ['@e' => $e->getMessage()]);
      return NULL;
    }
  }

  public function loadLocalCopress(string $file): Response {
    $module_path = \Drupal::service('extension.list.module')
      ->getPath('staticjson');

    // Ruta del ZIP
    $zip_path = DRUPAL_ROOT . '/' . $module_path . '/json/'  . $file . '.zip';

    if (!file_exists($zip_path)) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException(
        'El archivo ZIP no existe.'
      );
    }
  return new BinaryFileResponse(
    $zip_path,
    200,
    [
      'Content-Type' => 'application/zip',
      'Content-Disposition' => 'attachment; filename="' . basename($zip_path) . '"',
      'Content-Length' => filesize($zip_path),
      'Cache-Control' => 'private',
    ]
  );
  }
}
