<?php

namespace Drupal\staticjson\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\File\FileSystemInterface;

use ZipArchive;
use RecursiveIteratorIterator;
use RecursiveDirectoryIterator;


class StaticJsonFormHandler {

  protected ConfigFactoryInterface $configFactory;
  protected MessengerInterface $messenger;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger
  ) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
  }

  /**
   * Ejecuta la lógica al guardar entidad.
   */
  public function handleEntity(EntityInterface $entity, array $context = []): void {


    $entity_type = $context['entity_type'] ?? $entity->getEntityTypeId();
    $bundle = $context['bundle'] ?? (method_exists($entity, 'bundle') ? $entity->bundle() : NULL);
    $is_new = $context['is_new'] ?? $entity->isNew();
    $form_id = $context['form_id'] ?? 'unknown';

    $config = $this->configFactory->get('staticjson.settings');
    $selected_content_types = array_filter($config->get('content_types') ?? []);

    if (empty($selected_content_types)) {
      return;
    }

    static $lookup = NULL;
    if ($lookup === NULL) {
      $lookup = array_flip($selected_content_types);
    }

    if (!method_exists($entity, 'bundle')) {
      return;
    }

    if (!isset($lookup[$entity->bundle()])) {
      return;
    }

    // ---- TU LÓGICA ORIGINAL ----
    $json_data = $this->entityToJsonDeep($entity);
    $this->saveEntityJson($entity, $json_data);

    $this->messenger->addStatus('JSON estático generado.');
  }

  protected function entityToJsonDeep(EntityInterface $entity, int $depth = 0, array &$visited = []) {

    // -------------------------------
    // CONFIGURACIÓN SEGURA
    // -------------------------------
    $max_depth = 2; // limite QUEMADO
    $exclude_types = ['user', 'file']; // tipos excluidos
    $sensitive_fields = [
      'pass', 'password', 'mail', 'email', 'user_password',
      'authmap', 'roles', 'permissions', 'token', 'secret',
      'private_key', 'api_key', 'consumer_key', 'consumer_secret',
    ];

    $entity_type = $entity->getEntityTypeId();
    $entity_id   = $entity->id();

    // -------------------------------
    // EXCLUSIONES DE ENTIDADES COMPLETAS
    // -------------------------------
    if (in_array($entity_type, $exclude_types, TRUE)) {
      return [
        '_entity_type' => $entity_type,
        '_id' => $entity_id,
        '_excluded' => TRUE,
      ];
    }

    // -------------------------------
    // ANTI-RECURSIÓN (ciclos)
    // -------------------------------
    $key = "$entity_type:$entity_id";

    if (isset($visited[$key])) {
      return [
        '_entity_type' => $entity_type,
        '_id' => $entity_id,
        '_recursive_reference' => TRUE,
      ];
    }

    $visited[$key] = TRUE;

    // -------------------------------
    // LÍMITE DE PROFUNDIDAD
    // -------------------------------
    if ($depth >= $max_depth) {
      return [
        '_entity_type' => $entity_type,
        '_id' => $entity_id,
        '_depth_limit' => TRUE,
      ];
    }

    // -------------------------------
    // BASE JSON
    // -------------------------------
    $output = [
      '_entity_type' => $entity_type,
      '_bundle'      => method_exists($entity, 'bundle') ? $entity->bundle() : NULL,
      '_id'          => $entity_id,
    ];

    // -------------------------------
    // SI NO ES FIELDABLE → metadata
    // -------------------------------
    if (!$entity instanceof FieldableEntityInterface) {
      $output['_raw'] = $entity->toArray();
      return $output;
    }

    // -------------------------------
    // PROCESAR CAMPOS
    // -------------------------------
    foreach ($entity->getFields() as $field_name => $field) {

      // 1) Proteger campos sensibles
      if (in_array($field_name, $sensitive_fields, TRUE)) {
        $output[$field_name] = '[PROTEGIDO]';
        continue;
      }

      $definition = $field->getFieldDefinition();
      $type = $definition->getType();

      // --------------------------------
      // Caso 1: campo NO referencia
      // --------------------------------
      if ($type !== 'entity_reference' && $type !== 'entity_reference_revisions') {
        $output[$field_name] = $field->getValue();
        continue;
      }

      // --------------------------------
      // Caso 2: campo de referencia
      // --------------------------------
      $target_type = $definition->getSetting('target_type');

      if (!$target_type) {
        $output[$field_name] = $field->getValue();
        continue;
      }

      $items = [];
      foreach ($field->getValue() as $item) {

        if (!isset($item['target_id'])) {
          $items[] = $item;
          continue;
        }

        $target = \Drupal::entityTypeManager()
          ->getStorage($target_type)
          ->load($item['target_id']);

        if ($target instanceof EntityInterface) {
          // Recursión segura
          $items[] = $this->entityToJsonDeep($target, $depth + 1, $visited);
        }
        else {
          $items[] = $item;
        }
      }

      $output[$field_name] = $items;
    }

    return $output;
  }
  protected function saveEntityJson(EntityInterface $entity, array $json_data) {
    $bundle = $entity->bundle();
    $id = $entity->id();

    // Convertir a JSON
    $json_string = json_encode($json_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    // -----------------------------
    // Ruta del módulo
    // -----------------------------
    $module_path = \Drupal::service('extension.list.module')
        ->getPath('staticjson');

    // Base /json/
    $base_dir = DRUPAL_ROOT . '/' . $module_path . '/json/';

    // Carpeta por bundle SIEMPRE termina en "/"
    $target_dir = $base_dir . $bundle . '/';

    // Crear carpeta si no existe
    \Drupal::service('file_system')->prepareDirectory(
        $target_dir,
        FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    // -----------------------------
    // Guardar archivo principal por ID
    // -----------------------------
    $id_filepath = $target_dir .'/'. $id . '.json';
    file_put_contents($id_filepath, $json_string);

    // -----------------------------
    // Alias
    // -----------------------------
    $alias = NULL;

    if ($entity->hasLinkTemplate('canonical')) {
        $internal = $entity->toUrl('canonical')->getInternalPath();
        $alias_raw = \Drupal::service('path_alias.manager')->getAliasByPath('/' . $internal);

        if (!empty($alias_raw) && $alias_raw !== '/' . $internal) {
        $alias = ltrim($alias_raw, '/');
        $alias = str_replace('/', '_', $alias); // limpiar alias
        }
    }

    if ($alias) {
        $alias_filepath = $target_dir .'/'. $alias . '.json';
        file_put_contents($alias_filepath, $json_string);
    }
  }

public function generateCompress(string $Name) {
  $module_path = \Drupal::service('extension.list.module')
    ->getPath('staticjson');

  $base_dir = DRUPAL_ROOT . '/' . $module_path . '/json/';
  $zip_path = $base_dir . '/' . $Name . '.zip';

  // Validar que la carpeta exista
  if (!is_dir($base_dir)) {
    throw new \Exception('La carpeta base no existe: ' . $base_dir);
  }

  $zip = new ZipArchive();

  if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
    throw new \Exception('No se pudo crear el archivo ZIP');
  }

  $files = new RecursiveIteratorIterator(
    new RecursiveDirectoryIterator($base_dir, RecursiveDirectoryIterator::SKIP_DOTS),
    RecursiveIteratorIterator::LEAVES_ONLY
  );

  foreach ($files as $file) {
    if (!$file->isDir()) {
      $filePath = $file->getRealPath();
      // Ruta relativa dentro del zip
      $relativePath = substr($filePath, strlen($base_dir));
      $zip->addFile($filePath, $relativePath);
    }
  }

  $zip->close();

  return $zip_path;
}

}
