<?php

namespace Drupal\status_pages\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;

/**
 * Controller for custom status pages.
 */
class StatusPagesController extends ControllerBase {

  /**
   * Displays the 403 Access Denied page.
   *
   * @return array
   *   Render array for the access denied page.
   */
  public function accessDenied(): array {
    return $this->buildComponent('403');
  }

  /**
   * Displays the 404 Not Found page.
   *
   * @return array
   *   Render array for the not found page.
   */
  public function notFound(): array {
    return $this->buildComponent('404');
  }

  /**
   * Builds the render array for a status page.
   *
   * @param string $status
   *   The status code (e.g., '403', '404').
   *
   * @return array
   *   Render array for the status page.
   */
  protected function buildComponent($status): array {
    $config = $this->config('status_pages.settings')->get($status) ?? [];
    return [
      '#theme' => 'status_page',
      '#title' => $config['title'] ?? NULL,
      '#subtitle' => $config['subtitle'] ?? NULL,
      '#description' => $config['description'] ?? NULL,
      '#link_text' => $config['link_text'] ?? NULL,
      '#link_url' => $this->getUrl($config['link_url']),
      '#status_code' => $status,
      '#attached' => [
        'library' => ['status_pages/status-page'],
      ],
    ];
  }

  /**
   * Returns a URL from a given URI.
   *
   * @param string $uri
   *   The URI to convert to a URL.
   *
   * @return string
   *   The URL as a string.
   */
  protected function getUrl($uri): string {
    try {
      return Url::fromUri($uri)->toString();
    }
    catch (\Exception $e) {
      // If the URI is invalid, return the front page URL.
      return Url::fromRoute('<front>')->toString();
    }
  }

}
