<?php

declare(strict_types=1);

namespace Drupal\status_pages\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure status pages settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'status_pages_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['status_pages.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('status_pages.settings');

    foreach (['403', '404'] as $status) {
      $status_config = $config->get($status) ?? [];
      $form[$status] = [
        '#type' => 'fieldset',
        '#title' => $this->t('@status Status Page', ['@status' => $status]),
        '#tree' => TRUE,
      ];

      $form[$status]['title'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Title'),
        '#reqired' => TRUE,
        '#default_value' => $status_config['title'] ?? NULL,
      ];

      $form[$status]['subtitle'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Subtitle'),
        '#default_value' => $status_config['subtitle'] ?? NULL,
      ];

      $form[$status]['description'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Description'),
        '#default_value' => $status_config['description'] ?? NULL,
      ];

      $form[$status]['link_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Link text'),
        '#default_value' => $status_config['link_text'] ?? NULL,
      ];

      $form[$status]['link_url'] = [
        '#type' => 'textfield',
        '#title' => $this->t('URL'),
        '#default_value' => static::getUriAsDisplayableString($status_config['link_url'] ?? ''),
        '#element_validate' => [
          ['Drupal\\link\\Plugin\\Field\\FieldWidget\\LinkWidget', 'validateUriElement'],
        ],
        '#states' => [
          'required' => [
            ':input[name="' . $status . '][link_text]"]' => ['filled' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $config = $this->config('status_pages.settings');
    foreach (['403', '404'] as $status) {
      if (!empty($values[$status])) {
        $config->set($status, $values[$status]);
      }
    }
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets the URI without the 'internal:' or 'entity:' scheme.
   *
   * @param string $uri
   *   The URI to get the displayable string for.
   *
   * @return string
   *   The displayable string for the URI.
   *
   * @see static::getUserEnteredStringAsUri()
   */
  protected static function getUriAsDisplayableString($uri) {
    if (empty($uri)) {
      return $uri;
    }

    try {
      $scheme = parse_url($uri, PHP_URL_SCHEME);

      // By default, the displayable string is the URI.
      $displayable_string = $uri;

      // A different displayable string may be chosen in case of the 'internal:'
      // or 'entity:' built-in schemes.
      if ($scheme === 'internal') {
        $uri_reference = explode(':', $uri, 2)[1];
        $path = parse_url($uri, PHP_URL_PATH);
        if ($path === '/') {
          $uri_reference = '<front>' . substr($uri_reference, 1);
        }

        $displayable_string = $uri_reference;
      }
      elseif ($scheme === 'entity') {
        [$entity_type, $entity_id] = explode('/', substr($uri, 7), 2);
        if ($entity_type == 'node' && $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id)) {
          $displayable_string = EntityAutocomplete::getEntityLabels([$entity]);
        }
      }
      elseif ($scheme === 'route') {
        $displayable_string = ltrim($displayable_string, 'route:');
      }

      return $displayable_string;
    }
    catch (\InvalidArgumentException) {
      // If the URI is invalid, return it as is.
    }

    return $uri;
  }

}
