<?php

declare(strict_types=1);

namespace Drupal\status_pages\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Render\MainContent\MainContentRendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for custom status pages.
 */
final class StatusPagesController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * Constructs a new StatusPagesController object.
   *
   * @param \Drupal\Core\Render\MainContent\MainContentRendererInterface $htmlRenderer
   *   The HTML main content renderer service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route match.
   */
  public function __construct(
    private readonly MainContentRendererInterface $htmlRenderer,
    private readonly RequestStack $requestStack,
    private readonly RouteMatchInterface $routeMatch,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('main_content_renderer.html'),
      $container->get('request_stack'),
      $container->get('current_route_match'),
    );
  }

  /**
   * Renders the 403 Access Denied status page within the site layout.
   *
   * Sets the HTTP status code to 403 and applies an X-Robots-Tag header
   * to discourage indexing.
   *
   * @return \Drupal\Core\Render\HtmlResponse
   *   The rendered HTML response.
   */
  public function accessDenied(): HtmlResponse {
    $build = $this->buildComponent('403');

    // Render inside Drupal's full page (html/page templates, regions, libraries).
    $response = $this->htmlRenderer->renderResponse(
      $build,
      $this->requestStack->getCurrentRequest(),
      $this->routeMatch
    );

    $response->setStatusCode(403);

    $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

    return $response;
  }

  /**
   * Renders the 404 Not Found status page within the site layout.
   *
   * Sets the HTTP status code to 404 and applies an X-Robots-Tag header
   * to discourage indexing.
   *
   * @return \Drupal\Core\Render\HtmlResponse
   *   The rendered HTML response.
   */
  public function notFound(): HtmlResponse {
    $build = $this->buildComponent('404');

    $response = $this->htmlRenderer->renderResponse(
      $build,
      $this->requestStack->getCurrentRequest(),
      $this->routeMatch
    );

    $response->setStatusCode(404);
    $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

    return $response;
  }

  /**
   * Builds the render array for a status page.
   *
   * @param string $status
   *   The HTTP status code as a string (for example, '403' or '404').
   *
   * @return array
   *   A render array for the status page component.
   */
  protected function buildComponent(string $status): array {
    $config = $this->config('status_pages.settings')->get($status) ?? [];

    return [
      '#theme' => 'status_page',
      '#title' => $config['title'] ?? NULL,
      '#subtitle' => $config['subtitle'] ?? NULL,
      '#description' => $config['description'] ?? NULL,
      '#link_text' => $config['link_text'] ?? NULL,
      '#link_url' => $this->getUrl($config['link_url'] ?? null),
      '#status_code' => $status,
      '#attached' => [
        'library' => ['status_pages/status-page'],
      ],
      '#cache' => [
        'max-age' => 0,
        'tags' => ['config:status_pages.settings'],
      ],
    ];
  }

  /**
   * Converts a URI to an absolute URL string.
   *
   * Falls back to the front page if the URI is missing or invalid.
   *
   * @param string|null $uri
   *   The URI to convert to a URL, or NULL to use the front page.
   *
   * @return string
   *   The absolute URL as a string.
   */
  protected function getUrl(?string $uri): string {
    try {
      return $uri ? Url::fromUri($uri)->toString() : Url::fromRoute('<front>')->toString();
    }
    catch (\Throwable) {
      return Url::fromRoute('<front>')->toString();
    }
  }

}
