<?php

namespace Drupal\steam_login\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Steam OpenId' Block.
 *
 * @Block(
 *   id = "steam_openid",
 *   admin_label = @Translation("Steam OpenId"),
 * )
 */
class SteamOpenIdLogInOut extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The steam_login module path.
   *
   * @var string
   */
  protected $modulePath;

  /**
   * Steam OpenId block constructor.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleHandler $moduleHandler,
    protected AccountProxyInterface $account,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->modulePath = $moduleHandler->getModule('steam_login')->getPath();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('module_handler'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $form['steam_login_button'] = [
      '#type' => 'radios',
      '#options' => [
        0 => "<img src=\"/$this->modulePath/images/openid_01.png\" />",
        1 => "<img src=\"/$this->modulePath/images/openid_02.png\" />",
      ],
      '#default_value' => $this->configFactory->getEditable('steam_login.config')
        ->get('login_button'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configFactory->getEditable('steam_login.config')
      ->set('login_button', $form_state->getValue('steam_login_button'))
      ->save();

    parent::blockSubmit($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];

    if (!$this->account->isAuthenticated()) {
      switch ($this->configFactory->getEditable('steam_login.config')->get('login_button')) {
        case 0:
          $img = "<img src=\"/$this->modulePath/images/openid_01.png\" />";
          break;

        case 1:
          $img = "<img src=\"/$this->modulePath/images/openid_02.png\" />";
          break;

        default:
          return $build;
      }

      $build['steam_openid'] = [
        '#type' => 'link',
        '#title' => [
          '#markup' => $img,
        ],
        '#url' => Url::fromRoute('steam_login.openid'),
      ];

      return $build;
    }

    $build['account'] = [
      '#type' => 'link',
      '#title' => $this->account->getDisplayName(),
      '#url' => Url::fromRoute('entity.user.canonical', ['user' => $this->account->id()]),
    ];
    $build['separator'] = [
      '#markup' => ' | ',
    ];
    $build['logout'] = [
      '#type' => 'link',
      '#title' => 'Logout',
      '#url' => Url::fromRoute('user.logout'),
    ];

    return $build;

  }

}
