<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginTrait;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\stenographer\Attribute\StenographerAdapter;
use Drupal\stenographer\DataAdapterBase;
use Drupal\stenographer\Plugin\Derivative\EntityAdapterDeriver;
use Drupal\stenographer\TypedDataAdapterInterface;
use Drupal\user\EntityOwnerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Data adapter for retrieving data from an entity.
 */
#[StenographerAdapter(
  id: 'entity',
  deriver: EntityAdapterDeriver::class,
)]
class EntityAdapter extends DataAdapterBase implements TypedDataAdapterInterface, ContextAwarePluginInterface, ContainerFactoryPluginInterface {

  use ContextAwarePluginTrait;

  /**
   * Contains a data definition map for discovered entity types.
   *
   * @var array<string,array<string,\Drupal\Core\TypedData\DataDefinitionInterface[]>>
   */
  protected static array $entityProperties = [];

  /**
   * The entity type ID for the adapter that this data expects.
   *
   * @var string
   */
  protected string $entityTypeId;

  /**
   * Create a new instance of the current user data adapter.
   *
   * @param array $config
   *   The plugin configuration values.
   * @param string $pluginId
   *   The unique ID of the plugin.
   * @param mixed $definition
   *   The plugin definition from the plugin manager discovery.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $fieldManager
   *   Manager service for getting bundle field definitions for entities.
   * @param \Drupal\Core\TypedData\TypedDataManagerInterface $typedDataManager
   *   Manager for typed data definitions.
   */
  public function __construct(
    array $config,
    string $pluginId,
    $definition,
    protected EntityFieldManagerInterface $fieldManager,
    protected TypedDataManagerInterface $typedDataManager,
  ) {
    parent::__construct($config, $pluginId, $definition);

    // Capture the entity type ID for the plugin.
    [, $this->entityTypeId] = explode(':', $pluginId, 2);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_field.manager'),
      $container->get('typed_data_manager')
    );
  }

  /**
   * Fetches the entity to use as the data source for this adapter.
   *
   * @param array<string,mixed> $data
   *   Event context data to use when trying to extract entity values from.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The entity being worked on by this data adaptor.
   */
  protected function getEntity(array $data): EntityInterface {
    $entityKey = $this->configuration['entity_key'] ?? $this->entityTypeId;
    $entity = $data[$entityKey] ?? $data['entity'] ?? $this->getContextValue($entityKey);

    if ($entity instanceof EntityInterface && $entity->getEntityTypeId() === $this->entityTypeId) {
      return $entity;
    }

    $err = sprintf('Invalid context value. Expected entity of type %s but got %s', $this->entityTypeId, $entity->getEntityTypeId());
    throw new ContextException($err);
  }

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $properties = [];

    $entity = $this->getEntity($data);
    $entityType = $entity->getEntityType();
    $bundle = $entity->bundle();

    if (!isset(self::$entityProperties[$entityType->id()][$bundle])) {
      $fieldDefinitions = $this->fieldManager->getFieldDefinitions($this->entityTypeId, $bundle);
      foreach ($fieldDefinitions as $fieldDef) {
        $storageDef = $fieldDef->getFieldStorageDefinition();
        $fieldPropName = $storageDef->getMainPropertyName();
        $properties[$fieldDef->getName()] = $storageDef->getPropertyDefinition($fieldPropName);
      }

      $properties['@id'] = $properties[$entityType->getKey('id')];
      $properties['@entity_type'] = DataDefinition::create('string')
        ->setLabel('Entity type');

      if ($entityType->hasKey('bundle')) {
        $properties['@bundle'] = $properties[$entityType->getKey('bundle')] ?? DataDefinition::create('string');
      }

      if ($entity instanceof EntityOwnerInterface) {
        $data['@owner'] = DataDefinition::create('integer')
          ->setLabel('Owner');
      }

      self::$entityProperties[$entityType->id()][$bundle] = $properties;
    }

    return self::$entityProperties[$entityType->id()][$bundle];
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    $entity = $this->getEntity($data);

    return match($name) {
      '@entity_type' => TRUE,
      '@id' => TRUE,
      '@bundle' => $entity->getEntityType()->hasKey('bundle'),
      '@owner' => $entity instanceof EntityOwnerInterface,
      $entity instanceof FieldableEntityInterface => $entity->hasField($name),
      default => isset($this->propertyDefinitions($data)[$name]),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    $entity = $this->getEntity($data);

    switch ($name) {
      case '@id':
        return $entity->id();

      case '@bundle':
        return $entity->bundle();

      case '@entity_type':
        return $entity->getEntityTypeId();

      case '@owner':
        return $entity instanceof EntityOwnerInterface ? $entity->getOwnerId() : NULL;
    }

    try {
      if ($entity instanceof FieldableEntityInterface) {
        @[$fieldId, $property] = explode('.', $name, 2);
        $field = $entity->get($fieldId);

        if ($field && !$field->isEmpty()) {
          $valueProperty = $property ?: (get_class($field->first()) . '::mainPropertyName')();

          if (!empty($valueProperty)) {
            if ($field->count() > 1) {
              $values = [];
              foreach ($field as $item) {
                $values[] = $item->{$valueProperty};
              }

              return $values;
            }

            return $field->first()?->{$valueProperty};
          }
        }
      }
    }
    catch (\InvalidArgumentException $e) {
      // Field does not exists for this entity.
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function applyData(array &$data, array $src, array $properties): void {
    parent::applyData($data, $src, $properties);

    if ($this->configuration['includeType'] ?? TRUE) {
      $entity = $this->getEntity($src);
      $data['targetType'] = $this->entityTypeId;
      $data['targetId'] = $entity->id();

      if ($entity instanceof EntityOwnerInterface) {
        $data['owner'] = $entity->getOwnerId();
      }
    }
  }

}
