<?php

namespace Drupal\stenographer;

use Drupal\Component\Discovery\DiscoverableInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\Trigger\TriggerCollection;
use Drupal\toolshed\Discovery\YamlDiscovery;
use Drupal\toolshed\Strategy\StrategyDefinitionInterface;
use Drupal\toolshed\Strategy\StrategyManager;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Manager for tracking handlers that are defined from YAML definition files.
 */
class RecorderManager extends StrategyManager implements RecorderManagerInterface {

  use LoggerChannelTrait;

  const DISCOVERY_NAME = 'stenographer';

  /**
   * Get all the defined and loaded recorder instances.
   *
   * @var \Drupal\stenographer\RecorderInterface[]|null
   */
  protected array|null $recorders = NULL;

  /**
   * Recorders indexed by trigger type.
   *
   * The trigger map just maps to the recorder IDs so it can be cached.
   *
   * @var array<string,\Drupal\stenographer\RecorderInterface[]>|null
   */
  protected ?array $byTrigger = NULL;

  /**
   * Create a new instance of the RecorderManager strategy manager class.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\stenographer\Trigger\TriggerCollection $triggers
   *   The Stenographer trigger collector. Builds and registers Stenograher
   *   trigger handlers.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend to store definitions and trigger maps.
   */
  public function __construct(
    ModuleHandlerInterface $moduleHandler,
    protected TriggerCollection $triggers,
    #[Autowire(service: 'cache.discovery')]
    CacheBackendInterface $cache,
  ) {
    parent::__construct(static::DISCOVERY_NAME, $moduleHandler, $cache, ['stenographer:recorders']);
  }

  /**
   * Create a recorder discovery instance to find recorder definitions.
   *
   * @param array $directories
   *   Directories to search for definitions. Typically this should be module
   *   root directories.
   *
   * @return \Drupal\Component\Discovery\DiscoverableInterface
   *   A discovery instance to search for recorder definitions.
   */
  public static function createDiscovery(array $directories): DiscoverableInterface {
    return new YamlDiscovery(static::DISCOVERY_NAME, $directories);
  }

  /**
   * {@inheritdoc}
   */
  public function getRecorders(?array $ids = NULL): array {
    if (!isset($this->recorders)) {
      $this->recorders = [];

      foreach ($this->getDefinitions() as $id => $definition) {
        try {
          /** @var \Drupal\stenographer\RecorderInterface $recorder */
          $recorder = $this->getInstance($definition->id());
          $this->recorders[$id] = $recorder;
        }
        catch (\Throwable $e) {
          $logger = $this->getLogger('stenographer');
          Error::logException($logger, $e);
        }
      }
    }

    if (isset($ids)) {
      $result = [];
      foreach ($ids as $id) {
        if (isset($this->recorders[$id])) {
          $result[$id] = $this->recorders[$id];
        }
      }

      return $result;
    }

    return $this->recorders;
  }

  /**
   * {@inheritdoc}
   */
  public function getByTrigger(string $type, ?string $trigger = NULL): array {
    if (!isset($this->byTrigger)) {
      $cid = 'stenographer:triggers';

      if ($cached = $this->cacheGet($cid)) {
        $this->byTrigger = $cached->data;
      }
      else {
        $this->byTrigger = [];
        foreach ($this->triggers->getIterator() as $builder) {
          $this->byTrigger[$builder->getTriggerType()] = [];
        }

        foreach ($this->triggers->getIterator() as $builder) {
          $type = $builder->getTriggerType();
          $this->byTrigger[$type] = [
            ...$this->byTrigger[$type],
            ...$builder->buildTriggers($this->getRecorders()),
          ];
        }
        $this->cacheSet($cid, $this->byTrigger);
      }
    }

    if ($trigger) {
      return $this->byTrigger[$type][$trigger] ?? [];
    }
    return $this->byTrigger[$type] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultStrategyClass(array $definition): string {
    $type = $definition['type'] ?? RecorderDefinition::defaults()['type'];

    // Use the Audit class when the event type is audit.
    return 'audit' === $type ? AuditRecorder::class : Recorder::class;
  }

  /**
   * {@inheritdoc}
   */
  protected function processDefinition(string $id, array $definition): StrategyDefinitionInterface {
    if (empty($definition['class'])) {
      $definition['class'] = $this->defaultStrategyClass($definition);
    }

    return new RecorderDefinition($id, $definition);
  }

  /**
   * {@inheritdoc}
   */
  protected function getDiscovery(): DiscoverableInterface {
    $dirs = $this->moduleHandler->getModuleDirectories();

    $discovery = static::createDiscovery($dirs);
    if ($discovery instanceof YamlDiscovery) {
      $discovery->addTranslatableProperty('label', 'label_context');
      $discovery->addTranslatableProperty('description', 'description_context');
    }

    return $discovery;
  }

}
