<?php

namespace Drupal\stenographer\Trigger;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\RecorderManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Service / utility for consistent logging of entity events.
 */
class EntityHandler {

  use LoggerChannelTrait;

  /**
   * Creates a new instance of the EntityLogger helper.
   *
   * @param \Drupal\stenographer\Trigger\TriggerBuilderInterface $triggers
   *   Trigger builder which creates the trigger definitions and mapping for
   *   the triggers which this handler executes.
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   The recorder strategy manager.
   */
  public function __construct(
    #[Autowire(service: 'stenographer.entity_triggers')]
    public readonly TriggerBuilderInterface $triggers,
    protected RecorderManagerInterface $recorderManager,
  ) {}

  /**
   * Record the entity events based on the entity action.
   *
   * @param string $action
   *   The entity operation and actions that is being logged.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The target entity for the event.
   */
  public function logEvents(string $action, EntityInterface $entity): void {
    $entityType = $entity->getEntityType();
    $entityTypeId = $entityType->id();
    $bundle = $entityType->hasKey('bundle') ? $entity->bundle() : NULL;

    $data['entity'] = $entity;
    $data[$entityTypeId] = $entity;
    $eventType = "{$entityTypeId}:{$action}";

    $ids = [];
    foreach ($this->getTriggers($action, $entityTypeId) as $trigger) {
      /** @var string[] $ids */
      $ids += $this->recorderManager->getByTrigger('entity', $trigger);
    }

    foreach (\array_unique($ids) as $loggerId) {
      try {
        $logger = $this->recorderManager->getInstance($loggerId);
        $definition = $logger->getDefinition();

        if ($bundle) {
          $bundles = $definition->getTriggers['entity'][$entityTypeId]['bundles'] ?? [];

          if ($bundles && !in_array($bundle, $bundles)) {
            continue;
          }
        }

        $logger->logEvent('entity', $eventType, $data);
      }
      catch (\Throwable $e) {
        $logger = $this->getLogger("stenographer:entity:{$entityTypeId}:{$action}");
        Error::logException($logger, $e);
      }
    }
  }

  /**
   * Get the event triggers that are mapped from the entity actions.
   *
   * @param string $action
   *   The entity operation / action which is triggering the event.
   * @param string $entityTypeId
   *   The entity type ID for the entity which is triggering this event.
   *
   * @return string[]
   *   A list of entity triggers to create log events for.
   */
  protected function getTriggers(string $action, string $entityTypeId): array {
    return match($action) {
      'create', 'insert' => [
        $entityTypeId . ':save',
        $entityTypeId . ':create',
      ],
      'update', 'edit' => [
        $entityTypeId . ':save',
        $entityTypeId . ':update',
      ],
      'delete', 'remove' => [
        $entityTypeId . ':delete',
      ],
      default => [],
    };
  }

}
