<?php

namespace Drupal\stenographer;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\toolshed\Strategy\ContainerInjectionStrategyInterface;
use Drupal\toolshed\Strategy\StrategyDefinitionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * The default audit implementation for Stenographer recorders.
 *
 * Recorders are the logging implementation of the Stenographer module. This
 * recorder class add audit data to all records to ensure that they provide
 * user information.
 *
 * They also can define the events that can trigger them.
 */
class AuditRecorder extends Recorder implements RecorderInterface, ContainerInjectionStrategyInterface {

  /**
   * The current user for this event.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The current HTTP request for the triggered log event.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $request;

  /**
   * The user session manager.
   *
   * @var \Drupal\Core\Session\SessionManagerInterface
   */
  protected SessionManagerInterface $sessionManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, string $id, StrategyDefinitionInterface $definition): static {
    /** @var \Symfony\Component\HttpFoundation\RequestStack $requestStack */
    $requestStack = $container->get('request_stack');

    // Add the services and data needed to append audit data logging events.
    /** @var static */
    return parent::create($container, $id, $definition)
      ->setRequest($requestStack->getCurrentRequest())
      ->setCurrentUser($container->get('current_user'))
      ->setSessionManager($container->get('session_manager'));
  }

  /**
   * Set the user to be used as the actor of the audit log records.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user for the audit log events.
   *
   * @return self
   *   The logger instance to use for method chaining.
   */
  public function setCurrentUser(AccountInterface $currentUser): self {
    $this->currentUser = $currentUser;
    return $this;
  }

  /**
   * Set the HTTP request to use when building audit events.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request to use building audit log events.
   *
   * @return self
   *   The logger instance to use for method chaining.
   */
  public function setRequest(Request $request): self {
    $this->request = $request;
    return $this;
  }

  /**
   * Sets the session manager service to get session identifiers.
   *
   * @param \Drupal\Core\Session\SessionManagerInterface $sessionManager
   *   The session manager service.
   *
   * @return self
   *   The logger instance to use for method chaining.
   */
  public function setSessionManager(SessionManagerInterface $sessionManager): self {
    $this->sessionManager = $sessionManager;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function logEvent(string $type, ?string $event = NULL, array $contexts = []): void {
    if ($this->shouldCapture()) {
      $data = $this->buildEventData($contexts) + [
        'actionId' => $event ?: $this->id(),
      ];

      $data['type'] = $this->definition->getType();
      if (!empty($event)) {
        $data['actionId'] = $event;
      }

      // Set the capture strategy for the storage API.
      $captureDef = $this->definition->getCapture();
      $config = is_array($captureDef) ? $captureDef : [];
      $data['capture'] = $this->capture->formatStrategy($config);

      // Ensure that all audit events include the info of the current user.
      $data['ipAddress'] = $this->request->getClientIp();
      $data['account'] = $this->currentUser->getDisplayName();
      $data['accountId'] = $this->currentUser->id();
      $data['timestamp'] = $this->time->getRequestTime();

      if (empty($data['session'])) {
        // Only apply the session for authenticated users.
        $data['session'] = $this->currentUser->isAuthenticated()
          ? Crypt::hashBase64($this->sessionManager->getId()) : NULL;
      }

      $this->sendEvents($data);
    }
  }

}
