<?php

namespace Drupal\stenographer\Capture;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\stenographer\Attribute\StenographerCapture;
use Drupal\toolshed\Strategy\ContainerInjectionStrategyInterface;
use Drupal\toolshed\Strategy\StrategyDefinitionInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Only capture this event based on the value of a user field.
 *
 * The logger needs to provide the capture defined with the configurations that
 * specify the field to check for values, and if the anonymous user is able to
 * trigger this event.
 *
 * @see self::shouldCapture()
 */
#[StenographerCapture(
  id: 'user_field',
  isPassThru: FALSE,
)]
class AccountFlagCapture extends EventCapture implements ContainerInjectionStrategyInterface {

  /**
   * The loaded user entity of the current user.
   *
   * @var \Drupal\user\UserInterface|null
   */
  protected ?UserInterface $currentUser;

  /**
   * Creates a new instance of the AccountFlagCapture strategy.
   *
   * @param string $id
   *   The unique strategy identifier.
   * @param \Drupal\stenographer\Capture\CaptureDefinition $definition
   *   The capture strategy definition and settings.
   * @param \Drupal\Core\Session\AccountInterface $currentAccount
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    string $id,
    CaptureDefinition $definition,
    AccountInterface $currentAccount,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($id, $definition);

    // Load the current user entity if authenticated. Anonymous users don't
    // get loaded since their fields are not relevant for checking.
    $this->currentUser = $currentAccount->isAuthenticated()
      ? $entityTypeManager->getStorage('user')->load($currentAccount->id())
      : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, string $id, StrategyDefinitionInterface $definition): static {
    return new static(
      $id,
      $definition,
      $container->get('current_user'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function shouldCapture(array $config = []): bool {
    if (!empty($config['field'])) {
      if ($user = $this->currentUser) {
        try {
          return $user->get($config['field'])->isEmpty();
        }
        catch (\InvalidArgumentException $e) {
          // Field does not exist, don't send the event, the send indicators
          // do not exist.
          return FALSE;
        }
      }
      else {
        // Should the anonymous user always get captured? Anonymous users don't
        // have valid fields to check for the capture status, but in some cases
        // will want to always get sent, but shouldn't be the default behavior.
        return !empty($config['captureAnonymous']);
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function formatStrategy(array $config = []): ?string {
    // The data is always captured on the storage API since the user flag
    // is being determined on the Drupal side.
    return 'always';
  }

}
