<?php

namespace Drupal\stenographer\Plugin\Stenographer\Storage;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Utility\Html;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\Attribute\StenographerStore;
use Drupal\stenographer\RecorderInterface;
use Drupal\stenographer\StoreInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Stenographer record storage using a Drupal watchdog.
 */
#[StenographerStore('drupal_watchdog')]
class WatchdogStore extends PluginBase implements StoreInterface, ContainerFactoryPluginInterface {

  /**
   * Create a new instance of a database storage logger.
   *
   * @param array $config
   *   Configuration array containing database table to log to.
   * @param string $plugin_id
   *   The unique plugin ID.
   * @param mixed $plugin_definition
   *   Definition of the plugin as discovered by the plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Open database connection for creating queries.
   */
  public function __construct(
    array $config,
    string $plugin_id,
    $plugin_definition,
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {
    parent::__construct($config, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $config, $plugin_id, $plugin_definition) {
    return new static(
      $config,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = [];
    $dependencies['module'][] = 'stenographer';

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'channel' => 'stenographer',
      'level' => 'info',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return $this->configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function write(array $data, RecorderInterface $recorder): void {
    $config = $this->getConfiguration();

    try {
      $message = Html::escape(var_export($data, TRUE));
      $message = nl2br($message);

      $this->loggerFactory->get($config['channel'])
        ->log($config['level'], $message);
    }
    catch (\Exception $e) {
      $logger = $this->loggerFactory->get('recorder');
      Error::logException($logger, $e);
    }
  }

}
