<?php

namespace Drupal\stenographer;

use Drupal\toolshed\Strategy\StrategyDefinitionBase;

/**
 * Define a Stenographer recorder.
 *
 * Recorders are the instances of the RecorderInterface whiche define what
 * event trigger the capture of data, the data which gets captured, and where
 * the data is captured to.
 */
class RecorderDefinition extends StrategyDefinitionBase {

  /**
   * The recorder type.
   *
   * Generally should be "activity" or "audit". Other types can be added, with
   * other types can be added using an event listener which maps the type to
   * a default class.
   *
   * @var string
   */
  public readonly string $type;

  /**
   * The default capture rate and strategy to use with this recorder.
   *
   * Recorders can override the capture strategy per trigger if a capture value
   * is provided with with the trigger definition.
   *
   * @var string|array
   */
  public readonly string|array $capture;

  /**
   * Information about hooks and events that will trigger sending of the event.
   *
   * Triggers get registered to hooks and event subscribers that listen for this
   * activity and attempt to fire when the event occurs.
   *
   * @var array
   */
  protected array $triggers;

  /**
   * List of data properties to include in the recorded event.
   *
   * The data properties are an array keyed by the data adapter ID, which will
   * extract and provide the data properties from services and context values.
   *
   * @var array
   */
  protected array $data;

  /**
   * The list of storage configs to setup destinations for recording events.
   *
   * @var array
   */
  protected array $storage;

  /**
   * Creates a new instance of the RecorderDefinition type.
   *
   * @param string $id
   *   The recorder definition identifier.
   * @param mixed[] $definition
   *   The recorder definition found from the discovery.
   */
  public function __construct(string $id, array $definition) {
    parent::__construct($id, $definition);

    if (!is_array($definition['storage'])) {
      $storageId = (string) $definition['storage'];
      $definition['storage'] = [
        $storageId => ['id' => $storageId],
      ];
    }

    [
      'type' => $this->type,
      'capture' => $this->capture,
      'trigger' => $this->triggers,
      'data' => $this->data,
      'storage' => $this->storage,
    ] = $definition + static::defaults();
  }

  /**
   * Get the default recorder values.
   *
   * @return array
   *   Default values to apply to the recorder definition if they are empty.
   */
  public static function defaults(): array {
    return [
      'type' => 'activity',
      'capture' => 'always',
    ];
  }

  /**
   * Get the recorder type.
   *
   * @return string
   *   The recorder type.
   */
  public function getType(): string {
    return $this->type;
  }

  /**
   * Get the capture definition.
   *
   * The capture can be just a string when the capture strategy doesn't need
   * additional configurations, but if additional options are supplied, this
   * value should be an array with a key "id" to indicate which strategy
   * is being used.
   *
   * @return string|array
   *   The capture settings defined as part of the definition.
   */
  public function getCapture(): string|array {
    return $this->capture;
  }

  /**
   * Get the definition of recorder event triggers.
   *
   * The trigger definition is used by Drupal hooks and event subscribers to
   * determine which recorders should be used when an event is triggered.
   *
   * @return array
   *   Get the trigger definitions which this recorder should react to.
   */
  public function getTriggers(): array {
    return $this->triggers;
  }

  /**
   * Provides the list of properties that should be included in the event data.
   *
   * The data definitions, are keyed by the data adapter and values are the
   * list of properties that should be included from that data adapter.
   *
   * Data adapters are responsible for pulling the data from contexts and
   * services.
   *
   * @return array
   *   List of data properties that should be included in the event data.
   */
  public function getDataDefinitions(): array {
    return $this->data;
  }

  /**
   * Destinations to send the events to when triggered.
   *
   * Recorder storage are plugins that can take the event data, format it, and
   * send it to be captured in their respective systems. A recorder can have
   * multiple storage destinations defined.
   *
   * Storage definitions are an array defined with the keys for storage plugin
   * ID (key: 'id') and plugin configuration (key: 'config').
   *
   * @return array
   *   A list of recorder event storage destination definitions.
   */
  public function getStorageDefinitions(): array {
    return $this->storage;
  }

}
