<?php

namespace Drupal\stenographer\Trigger;

use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\RecorderManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\File\Exception\AccessDeniedException;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Helper function for modifying and attaching tracking events to exceptions.
 */
class ExceptionHandler {

  use LoggerChannelTrait;

  /**
   * Create a new instance of the FormEvents helper.
   *
   * @param \Drupal\stenographer\Trigger\TriggerBuilderInterface $triggers
   *   Trigger builder which creates the trigger definitions and mapping for
   *   the triggers which this handler executes.
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   The recorder strategy manager.
   */
  public function __construct(
    #[Autowire(service: 'stenographer.exception_triggers')]
    public readonly TriggerBuilderInterface $triggers,
    protected RecorderManagerInterface $recorderManager,
  ) {}

  /**
   * Log exceptions with a stenographer recorder.
   *
   * @param \Throwable $exception
   *   Log an exception through the stenographer recorders.
   */
  public function logException(\Throwable $exception): void {
    $etype = NULL;
    if ($exception instanceof HttpException) {
      $etype = match (TRUE) {
        $exception instanceof AccessDeniedHttpException => 'access',
        $exception instanceof NotFoundHttpException => 'not_found',
        default => NULL,
      };
    }

    if (!$etype) {
      return;
    }

    $triggerType = $this->triggers->getTriggerType();
    foreach ($this->recorderManager->getByTrigger('exception', $etype) as $id) {
      try {
        $recorder = $this->recorderManager->getInstance($id);
        $eventId = $recorder->getTriggers($triggerType)[$etype] ?? "exception:{$etype}";
        $recorder->logEvent('exception', $eventId, ['exeption' => $exception]);
      }
      catch (\Throwable $e) {
        Error::logException($this->getLogger('stenographer'), $e);
      }
    }
  }

}
