<?php

namespace Drupal\stenographer;

use Drupal\toolshed\Strategy\ContainerInjectionStrategyInterface;

/**
 * Record events that are determined as auditable or activity based on actor.
 *
 * If the user acting on a target resource is not the owner or is not altering
 * their account, the event recorder will be forced as an audit activity. This
 * is meant to handle the case where an a alters entities owned
 * by other users - including profile, user fields, favorites, etc...
 */
class AuditAdminRecorder extends AuditRecorder implements RecorderInterface, ContainerInjectionStrategyInterface {

  /**
   * {@inheritdoc}
   */
  public function logEvent(string $type, ?string $event = NULL, array $data = []): void {
    if ($this->shouldCapture()) {
      // Set the capture strategy for the storage API.
      $captureDef = $this->definition->getCapture();
      $config = is_array($captureDef) ? $captureDef : [];
      $values = [
        'type' => $this->definition->getType(),
        'actionId' => $event ?: $this->id(),
        'capture' => $this->capture->formatStrategy($config),
      ] + $this->buildEventData($data);

      if ('audit' !== $values['type']) {
        $accountId = $this->currentUser->id();

        // Force all admin actions when the current user does not match the
        // owner or the account being updated to be an audit record.
        if (('user' === $values['targetType'] && $accountId != $values['targetId']) || ($values['owner'] && $accountId != $values['owner'])) {
          $values['type'] = 'audit';
          $values = $this->applyAuditProperties($values);
        }
      }

      $this->sendEvents($values);
    }
  }

}
