<?php

namespace Drupal\stenographer\EventSubscriber;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\stenographer\Hook\HookHandler;
use Drupal\stenographer\RecorderManagerInterface;
use Drupal\stenographer\Trigger\ExceptionHandler;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Event subscriber to allow trackers to capture Kernel events and errors.
 */
class RecorderEventSubscriber implements EventSubscriberInterface {

  use LoggerChannelTrait;

  /**
   * Create a new instance of the Stenographer RecorderEventSubscriber.
   *
   * Listens for exception that are compatible for logging to an enabled
   * Stenographer recorder service.
   *
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   Manager for Stenographer recorder definitions and instances.
   * @param \Symfony\Component\EventDispatcher\EventDispatcher $eventDispatcher
   *   The event dispatcher service.
   * @param \Drupal\stenographer\Trigger\ExceptionHandler $exceptTriggers
   *   Trigger manager for handling exceptions.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The hook discovery cache backend.
   */
  public function __construct(
    protected RecorderManagerInterface $recorderManager,
    protected EventDispatcherInterface $eventDispatcher,
    protected ExceptionHandler $exceptTriggers,
    #[Autowire(service: 'cache.discovery')]
    protected CacheBackendInterface $cache,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events = [];
    $events[KernelEvents::REQUEST][] = ['registerHooks', 100];
    $events[KernelEvents::TERMINATE][] = ['closeTrackers', -10];
    $events[KernelEvents::EXCEPTION][] = ['captureException', 100];

    return $events;
  }

  /**
   * Let trackers to close buffers when the request is completed and finalized.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   Event information about the request and response.
   */
  public function closeTrackers(TerminateEvent $event) {
    foreach ($this->recorderManager->getRecorders() as $instance) {
      $instance->close();
    }
  }

  /**
   * Allow recorders to log based on exceptions.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   Exception event generated from the Kernel for uncaught exceptions.
   */
  public function captureException(ExceptionEvent $event) {
    $this->exceptTriggers->logException($event->getThrowable());
  }

  /**
   * Register Stenographer hook methods at the start of the request.
   */
  public function registerHooks(): void {
    $cid = 'stenographer:hooks';
    $hooks = [];

    if ($cached = $this->cache->get($cid)) {
      $hooks = $cached->data;
    }
    else {
      /** @var \Drupal\stenographer\RecorderDefinition $definition */
      foreach ($this->recorderManager->getDefinitions() as $definition) {
        $keys = \array_keys($definition->getTriggers()['hook'] ?? []);
        $hooks += \array_combine($keys, $keys);
      }

      $this->cache->set($cid, $hooks, Cache::PERMANENT, [$cid]);
    }

    if ($hooks && $this->eventDispatcher instanceof EventDispatcher) {
      foreach ($hooks as $hook) {
        $this->eventDispatcher->addListener('drupal_hook.' . $hook, [
          new HookHandler($hook, $this->recorderManager),
          '__invoke',
        ], -100);
      }
    }
  }

}
