<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\stenographer\Attribute\StenographerAdapter;
use Drupal\stenographer\DataAdapterBase;
use Drupal\stenographer\TypedDataAdapterInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Data adapter for exposing data from the current active user.
 */
#[StenographerAdapter('current_user')]
class CurrentUserAdapter extends DataAdapterBase implements TypedDataAdapterInterface, ContainerFactoryPluginInterface {

  /**
   * Entity type definition from the EntityTypeManager::getDefinition().
   *
   * @var \Drupal\Core\Entity\ContentEntityTypeInterface|null
   */
  protected ?ContentEntityTypeInterface $entityDef = NULL;

  /**
   * Currently active user account loaded. Can contain anonymous uer.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $currentUser;

  /**
   * Create a new instance of the current user data adapter.
   *
   * @param array $config
   *   The plugin configuration values.
   * @param string $plugin_id
   *   The unique ID of the plugin.
   * @param mixed $plugin_definition
   *   The plugin definition from the plugin manager discovery.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   Account proxy of the current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service for getting entity definitions and
   *   entity handler classes.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $fieldManager
   *   Manager service for getting bundle field definitions for entities.
   */
  public function __construct(
    array $config,
    string $plugin_id,
    $plugin_definition,
    AccountProxyInterface $account,
    EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $fieldManager,
  ) {
    parent::__construct($config, $plugin_id, $plugin_definition);

    /** @var \Drupal\Core\Entity\ContentEntityType $entityType */
    $entityType = $entityTypeManager->getDefinition('user');
    $this->entityDef = $entityType;
    $this->currentUser = $entityTypeManager->getStorage('user')->load($account->id());
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $bundle = $this->currentUser->bundle();

    $properties = [];
    foreach ($this->fieldManager->getFieldDefinitions('user', $bundle) as $fieldDef) {
      $storageDef = $fieldDef->getFieldStorageDefinition();
      /** @var class-string<\Drupal\Core\Field\FieldItemBase> $fieldClass */
      $fieldClass = $fieldDef->getClass();
      $fieldPropName = ($fieldClass . '::mainPropertyName')();
      $properties[$fieldDef->getName()] = $fieldClass::propertyDefinitions($storageDef)[$fieldPropName];
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    return $this->currentUser->hasField($name);
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    if ($this->hasProperty($name, $data) && !$this->currentUser->{$name}->isEmpty()) {
      $field = $this->currentUser->{$name}->first();
      $mainProperty = (get_class($field) . '::mainPropertyName')();

      return !empty($mainProperty) ? $field->{$mainProperty} : NULL;
    }

    return NULL;
  }

}
