<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginTrait;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\stenographer\Attribute\StenographerAdapter;
use Drupal\stenographer\DataAdapterBase;
use Drupal\stenographer\Plugin\Derivative\EntityAdapterDeriver;
use Drupal\stenographer\TypedDataAdapterInterface;
use Drupal\user\EntityOwnerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Data adapter for retrieving data from an entity.
 */
#[StenographerAdapter(
  id: 'entity',
  deriver: EntityAdapterDeriver::class,
)]
class EntityAdapter extends DataAdapterBase implements TypedDataAdapterInterface, ContextAwarePluginInterface, ContainerFactoryPluginInterface {

  use ContextAwarePluginTrait;

  /**
   * Contains a data definition map for discovered entity types.
   *
   * @var array<string,array<string,\Drupal\Core\TypedData\DataDefinitionInterface[]>>
   */
  protected static array $entityProperties = [];

  /**
   * The entity type ID for the adapter that this data expects.
   *
   * @var string
   */
  protected string $entityTypeId;

  /**
   * Create a new instance of the current user data adapter.
   *
   * @param array $config
   *   The plugin configuration values.
   * @param string $pluginId
   *   The unique ID of the plugin.
   * @param mixed $definition
   *   The plugin definition from the plugin manager discovery.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $fieldManager
   *   Manager service for getting bundle field definitions for entities.
   * @param \Drupal\Core\TypedData\TypedDataManagerInterface $typedDataManager
   *   Manager for typed data definitions.
   */
  public function __construct(
    array $config,
    string $pluginId,
    $definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $fieldManager,
    protected TypedDataManagerInterface $typedDataManager,
  ) {
    parent::__construct($config, $pluginId, $definition);

    // Capture the entity type ID for the plugin.
    [, $this->entityTypeId] = explode(':', $pluginId, 2);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('typed_data_manager')
    );
  }

  /**
   * Fetches the entity to use as the data source for this adapter.
   *
   * @param array<string,mixed> $data
   *   Event context data to use when trying to extract entity values from.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The entity being worked on by this data adaptor.
   */
  protected function getEntity(array $data): EntityInterface {
    $entityKey = $this->configuration['entity_key'] ?? $this->entityTypeId;
    $entity = $data[$entityKey] ?? $data['entity'] ?? $this->getContextValue($entityKey);

    if (is_array($entity) && $entity['entity']) {
      $entity = $entity['entity'];
    }

    if ($entity instanceof EntityInterface && $entity->getEntityTypeId() === $this->entityTypeId) {
      return $entity;
    }

    $err = sprintf('Invalid context value. Expected entity of type %s but got %s', $this->entityTypeId, $entity->getEntityTypeId());
    throw new ContextException($err);
  }

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $properties = [];

    $entity = $this->getEntity($data);
    $entityType = $entity->getEntityType();
    $bundle = $entity->bundle();

    if (!isset(self::$entityProperties[$entityType->id()][$bundle])) {
      $fieldDefinitions = $this->fieldManager->getFieldDefinitions($this->entityTypeId, $bundle);
      foreach ($fieldDefinitions as $fieldDef) {
        $storageDef = $fieldDef->getFieldStorageDefinition();
        $fieldPropName = $storageDef->getMainPropertyName();
        $properties[$fieldDef->getName()] = $storageDef->getPropertyDefinition($fieldPropName);
      }

      $properties['@id'] = $properties[$entityType->getKey('id')];
      $properties['@label'] = DataDefinition::create('string')->setLabel('Label');
      $properties['@entity_type'] = DataDefinition::create('string')->setLabel('Entity type');

      if ($entityType->hasKey('bundle')) {
        $properties['@bundle'] = $properties[$entityType->getKey('bundle')] ?? DataDefinition::create('string');
      }

      if ($entity instanceof EntityOwnerInterface) {
        $data['@owner'] = DataDefinition::create('integer')
          ->setLabel('Owner');
      }

      self::$entityProperties[$entityType->id()][$bundle] = $properties;
    }

    return self::$entityProperties[$entityType->id()][$bundle];
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    $entity = $this->getEntity($data);

    return match($name) {
      '@entity_type', '@id', '@label' => TRUE,
      '@bundle' => $entity->getEntityType()->hasKey('bundle'),
      '@owner' => $entity instanceof EntityOwnerInterface,
      '@changeset' => $entity instanceof FieldableEntityInterface,
      default => $entity instanceof FieldableEntityInterface
        ? $entity->hasField($name)
        : isset($this->propertyDefinitions($data)[$name]),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    $entity = $this->getEntity($data);

    switch ($name) {
      case '@id':
        return $entity->id();

      case '@label':
        return $entity->label();

      case '@bundle':
        return $entity->bundle();

      case '@entity_type':
        return $entity->getEntityTypeId();

      case '@changeset':
        return $entity instanceof FieldableEntityInterface
          ? $this->generateChangeset($entity) : [];

      case '@owner':
        return $entity instanceof EntityOwnerInterface ? $entity->getOwnerId() : NULL;
    }

    try {
      if ($entity instanceof FieldableEntityInterface) {
        @[$fieldId, $property] = explode('.', $name, 2);
        $field = $entity->get($fieldId);

        if ($field && !$field->isEmpty()) {
          $valueProperty = $property ?: (get_class($field->first()) . '::mainPropertyName')();

          if (!empty($valueProperty)) {
            if ($field->count() > 1) {
              $values = [];
              foreach ($field as $item) {
                $values[] = $item->{$valueProperty};
              }

              return $values;
            }

            return $field->first()?->{$valueProperty};
          }
        }
      }
    }
    catch (\InvalidArgumentException $e) {
      // Field does not exists for this entity.
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function applyData(array &$data, array $src, array $properties): void {
    parent::applyData($data, $src, $properties);

    if ($this->configuration['includeType'] ?? TRUE) {
      $entity = $this->getEntity($src);
      $data['targetType'] = $this->entityTypeId;
      $data['targetId'] = $entity->id();

      if ($entity instanceof EntityOwnerInterface) {
        $data['owner'] = $entity->getOwnerId();
      }
    }
  }

  /**
   * Determine the names of entity fields which have changed.
   *
   * This method uses the "changesetInclude" and "changesetExclude" plugin
   * configuration to determine which fields to include as part of the
   * comparison for changes.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to generate a changeset from.
   *
   * @return string[]
   *   A list of field names which have changed if any were updated. Can also
   *   be empty if an original entity value was not available.
   */
  protected function generateChangeset(FieldableEntityInterface $entity): array {
    $original = $entity->original ?? NULL;

    if ($entity->isNew() || !$original instanceof FieldableEntityInterface) {
      // No changes reported for new entities or ones that do not provide an
      // entity to compare against.
      return [];
    }

    @[
      'changesetInclude' => $include,
      'changesetExclude' => $exclude,
    ] = $this->configuration['changesetInclude'];

    // Configuration specifies an inclusive changeset to be computed, only the
    // listed fields will be computed for changes.
    if ($include && is_array($include)) {
      $fields = \array_intersect_key($entity->getFields(), \array_flip($include));
    }
    else {
      $exclude = \array_fill_keys(is_array($exclude) ? $exclude : [], TRUE);
      $exclude['pass'] = TRUE;
      $exclude['changed'] = TRUE;

      // Remove these entity key values as they should not generally change or
      // be tracked as part of these changesets.
      $entityType = $entity->getEntityType();
      foreach (['id', 'revision', 'bundle', 'langcode', 'uuid'] as $key) {
        if ($entityType->hasKey($key)) {
          $exclude[$entityType->getKey($key)] = TRUE;
        }
      }

      $fields = \array_diff_key($entity->getFields(), $exclude);
    }

    $changedFields = [];
    foreach ($fields as $fieldName => $items) {
      $origItems = $original->get($fieldName);

      if (!$items->equals($origItems)) {
        $changedFields[] = $fieldName;
      }
    }

    return $changedFields;
  }

}
