<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\stenographer\Attribute\StenographerAdapter;
use Drupal\stenographer\DataAdapterBase;
use Drupal\stenographer\TypedDataAdapterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Data adapter for exposing data from the current HTTP Request.
 */
#[StenographerAdapter('http_request')]
class RequestAdapter extends DataAdapterBase implements TypedDataAdapterInterface, ContainerFactoryPluginInterface {

  /**
   * Create a new instance of the data adapter for exposing Request information.
   *
   * @param array $config
   *   Configuration for this data request adapter.
   * @param string $pluginId
   *   Unique identifier for the plugin manager.
   * @param mixed $pluginDefinition
   *   Plugin definition from discovery.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request being fulfilled.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The Drupal time service.
   */
  public function __construct(
    array $config,
    string $pluginId,
    $pluginDefinition,
    protected Request $request,
    protected TimeInterface $time,
  ) {
    parent::__construct($config, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('datetime.time')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    static $properties;

    if (!isset($properties)) {
      $properties['host'] = DataDefinition::create('string')
        ->setLabel('Server host');

      $properties['ip_address'] = DataDefinition::create('string')
        ->setLabel('Client IP address');

      $properties['method'] = DataDefinition::create('string')
        ->setLabel('HTTP request method');

      $properties['uri'] = DataDefinition::create('string')
        ->setLabel('Request URI');

      $properties['request_time'] = DataDefinition::create('timestamp')
        ->setLabel('Request time');
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $property, array $data): bool {
    $properties = $this->propertyDefinitions($data);

    return isset($properties[$property]);
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    $method = 'get' . static::toMethodCase($name);

    return $this->hasProperty($name, $data) && method_exists($this, $method)
      ? $this->{$method}()
      : NULL;
  }

  /**
   * Helper function to camel case strings.
   *
   * @param string $name
   *   The string name of the data to transform.
   *
   * @return string
   *   The transformed text, with the word capitalized.
   */
  protected static function toMethodCase(string $name): string {
    return preg_replace_callback('#(?:^|_)([a-z])#', function (array $matches) {
      return ucfirst($matches[1]);
    }, $name);
  }

  /**
   * Get the server host information from the current request.
   *
   * @return string
   *   Return the server host to which the request was made.
   */
  protected function getHost(): string {
    return $this->request->getHost();
  }

  /**
   * Get the HTTP request method for the current HTTP request.
   *
   * @return string
   *   The method of the HTTP request used (typically GET, POST, PUT or HEAD).
   */
  protected function getMethod(): string {
    return $this->request->getMethod();
  }

  /**
   * The client IP address from which the request came.
   *
   * @return string
   *   The IP address of the client.
   */
  protected function getIpAddress(): string {
    return $this->request->getClientIp();
  }

  /**
   * The request URI (path and query).
   *
   * @return string
   *   The request URI which includes the path and any URL query.
   */
  protected function getUri(): string {
    return $this->request->getRequestUri();
  }

  /**
   * Get the system timestamp for of the request.
   *
   * @return int
   *   The timestamp of the current system call.
   */
  protected function getRequestTime(): int {
    return $this->time->getRequestTime();
  }

}
