<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\ListDataDefinition;
use Drupal\stenographer\TypedDataAdapterInterface;

/**
 * Data adapter for retrieving data from an entity.
 */
class UserEntityAdapter extends EntityAdapter implements TypedDataAdapterInterface {

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $properties = parent::propertyDefinitions($data);
    $properties['@roles_changed'] = ListDataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Roles changed'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    /** @var \Drupal\user\UserInterface $user */
    $user = $this->getEntity($data);

    return match($name) {
      '@entity_type', '@id', '@label' => TRUE,
      '@changeset', '@roles_changed'  => TRUE,
      default => $user->hasField($name),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    if ('@roles_changed' === $name) {
      /** @var \Drupal\user\UserInterface $user */
      $user = $this->getEntity($data);

      // If roles have been changed, a "+" is for roles that got added,
      // and "-" prefixes roles which have been removed.
      if ($original = $user->original ?? NULL) {
        $origRoles = $original->getRoles();
        $roles = $user->getRoles();

        $changed = [];
        foreach (array_diff($roles, $origRoles) as $role) {
          $changed[] = '+' . $role;
        }
        foreach (array_diff($origRoles, $roles) as $role) {
          $changed[] = '-' . $role;
        }

        return $changed;
      }
    }

    return parent::get($name, $data);
  }

}
