<?php

namespace Drupal\stenographer\Plugin\Stenographer\Condition;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Plugin\PluginBase;
use Drupal\stenographer\Attribute\StenographerCondition;
use Drupal\stenographer\ConditionInterface;

/**
 * Condition plugin to check if data properties exists.
 *
 * Only evaluates to true if the event data has values for the specified
 * event data properties. If the configuration "test" has a value of "any"
 * then the condition is true if ANY of the listed properties has a value,
 * otherwise, the conditions is only true when ALL properties exist.
 */
#[StenographerCondition(
  id: 'has_data',
)]
class HasDataCondition extends PluginBase implements ConditionInterface, ConfigurableInterface {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      // Test if "all" or "any" properties listed have values.
      'test' => 'any',
      'properties' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(array $data): bool {
    $config = $this->getConfiguration() + $this->defaultConfiguration();

    if (!$config['properties']) {
      return TRUE;
    }
    elseif ('all' === $config['test']) {
      // All properties must have a value.
      foreach ($this->configuration['properties'] as $name) {
        if (empty($data[$name])) {
          return FALSE;
        }
      }

      return TRUE;
    }
    else {
      // First non-empty property will evaluate to TRUE.
      foreach ($this->configuration['properties'] as $name) {
        if (!empty($data[$name])) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

}
