<?php

namespace Drupal\stenographer;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\Capture\CaptureInterface;
use Drupal\toolshed\Strategy\ContainerInjectionStrategyInterface;
use Drupal\toolshed\Strategy\StrategyBase;
use Drupal\toolshed\Strategy\StrategyDefinitionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The default implementation for Stenographer recorders.
 *
 * Recorders are the event logging implementation of the Stenographer module.
 * They define the triggers, data properties to capture, and the storage
 * targets.
 *
 * Stenographer will register these to hooks and event subscribers based on
 * which triggers they have been defined and the conditions provided to
 * trigger them.
 *
 * @property \Drupal\stenographer\RecorderDefinition $definition
 */
class Recorder extends StrategyBase implements RecorderInterface, ContainerInjectionStrategyInterface {

  use LoggerChannelTrait;

  /**
   * The capture strategy to determine if an event should be captured.
   *
   * @var \Drupal\stenographer\Capture\CaptureInterface
   */
  protected CaptureInterface $capture;

  /**
   * Addition capture context information.
   *
   * @var array
   */
  protected array $captureContext = [];

  /**
   * The data conditions to check before sending events.
   *
   * @var \Drupal\stenographer\ConditionInterface[]
   */
  protected array $conditions;

  /**
   * An array of loaded data adapters.
   *
   * @var \Drupal\stenographer\DataAdapterInterface[]
   */
  protected array $adapters = [];

  /**
   * Storage handlers for the tracker to log data to.
   *
   * @var \Drupal\stenographer\StoreInterface[]
   */
  protected array $storage = [];

  /**
   * Create a new instance of the basic Stenographer recorder.
   *
   * @param string $id
   *   The unique string ID of the tracker.
   * @param \Drupal\stenographer\RecorderDefinition $definition
   *   The logger definition discovered from the YAML file.
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $contextHandler
   *   The context handler instance for applying contexts to plugins.
   * @param \Drupal\stenographer\CaptureStrategyManagerInterface $captureManager
   *   The capture manager service.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $conditionManager
   *   Plugin manager for data condition plugin discovery and creation.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $storageManager
   *   Plugin manager for tracker storage plugin discovery and creation.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $adapterManager
   *   Plugin manager for data adapter discovery and creation.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The datetime time service.
   */
  public function __construct(
    string $id,
    RecorderDefinition $definition,
    protected ContextHandlerInterface $contextHandler,
    CaptureStrategyManagerInterface $captureManager,
    PluginManagerInterface $conditionManager,
    PluginManagerInterface $storageManager,
    protected PluginManagerInterface $adapterManager,
    protected TimeInterface $time,
  ) {
    parent::__construct($id, $definition);

    $capture = $definition->getCapture();
    $capture = is_array($capture) ? $capture : ['id' => $capture];
    $this->capture = $captureManager->getInstance($capture['id']);
    $this->captureContext = $capture;

    $this->conditions = [];
    foreach ($definition->getConditionDefinitions() as $conditionDef) {
      $this->conditions[] = $conditionManager->createInstance(
        $conditionDef['id'],
        $conditionDef['configuration'] ?? [],
      );
    }

    // Create the storage definition for creating persisting logging info.
    foreach ($definition->getStorageDefinitions() as $storageDef) {
      $this->storage[] = $storageManager->createInstance(
        $storageDef['id'],
        $storageDef['configuration'] ?? []
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, string $id, StrategyDefinitionInterface $definition): static {
    return new static(
      $id,
      $definition,
      $container->get('context.handler'),
      $container->get('strategy.manager.stenographer.capture'),
      $container->get('plugin.manager.stenographer.condition'),
      $container->get('plugin.manager.stenographer.storage'),
      $container->get('plugin.manager.stenographer.data_adapter'),
      $container->get('datetime.time')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isActive(): bool {
    return $this->storage && $this->hasTriggers();
  }

  /**
   * {@inheritdoc}
   */
  public function hasTriggers(): bool {
    return (bool) $this->definition->getTriggers();
  }

  /**
   * {@inheritdoc}
   */
  public function getTriggers(?string $type = NULL): array {
    $triggers = $this->definition->getTriggers();

    return $type ? ($triggers[$type] ?? []) : $triggers;
  }

  /**
   * Fetch the data adapter for the plugin ID specified.
   *
   * @param string $adapterId
   *   The plugin ID of the data adapter to fetch the data for.
   * @param array $config
   *   The data adapter configurations.
   * @param \Drupal\Core\Plugin\Context\ContextInterface[] $contexts
   *   The contexts that can be applied to the data adapter instance.
   *
   * @return \Drupal\stenographer\DataAdapterInterface|null
   *   The loaded data adapter instance, matching the ID.
   */
  protected function fetchDataAdapter(string $adapterId, array $config = [], array $contexts = []): ?DataAdapterInterface {
    if (!isset($this->adapters[$adapterId])) {
      try {
        $adapter = $this->adapterManager->createInstance($adapterId, $config);
        $this->adapters[$adapterId] = $adapter ?: FALSE;

        if ($adapter instanceof ContextAwarePluginInterface) {
          $this->contextHandler->applyContextMapping($adapter, $contexts);
        }
      }
      catch (PluginException | ContextException $e) {
        $this->adapters[$adapterId] = FALSE;
      }
    }

    return $this->adapters[$adapterId] ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function logEvent(string $type, ?string $event = NULL, array $data = []): void {
    if ($this->shouldCapture()) {
      $values = [
        'type' => $this->definition->getType(),
        'actionId' => $event ?: $this->id(),
        'capture' => $this->capture->formatStrategy($this->captureContext),
        'timestamp' => $this->time->getRequestTime(),
      ] + $this->buildEventData($data);

      $this->sendEvents($values);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function close(): void {
    foreach ($this->storage as $store) {
      if ($store instanceof BufferedStoreInterface) {
        $store->flush();
      }
    }
  }

  /**
   * Based on the capture strategy, should this event be captured?
   *
   * @return bool
   *   TRUE if the event should be capture and logged.
   */
  protected function shouldCapture(): bool {
    return $this->capture->shouldCapture($this->captureContext);
  }

  /**
   * Determine if the event should be sent to the storage destinations.
   *
   * This is a chance for the recorder to determine if it should send the event
   * after the data is available. Returning FALSE will prevent the event from
   * being sent.
   *
   * @param array $data
   *   The event data to evaluate if the event should be sent.
   *
   * @return bool
   *   TRUE if the event should be sent, and FALSE if it should be prevented
   *   from being sent.
   */
  protected function shouldSend(array $data): bool {
    // First failed condition prevents the event from being sent.
    foreach ($this->conditions as $condition) {
      if (!$condition->evaluate($data)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Builds the event from the logger data definition.
   *
   * @param array $data
   *   Data to send to the data adapter.
   *
   * @return array
   *   The data prepared from the adapter definitions and the event storage.
   */
  protected function buildEventData(array $data): array {
    $values = [];
    foreach ($this->definition->getDataDefinitions() as $adapterId => $info) {
      if ($adapter = $this->fetchDataAdapter($adapterId, $info['config'] ?? [])) {
        $adapter->applyData($values, $data, $info['properties']);
      }
    }

    return $values;
  }

  /**
   * Send events to get recorded by the configured storage endpoints.
   *
   * @param array $data
   *   The data to write to the logging storage.
   */
  protected function sendEvents(array $data): void {
    if ($this->shouldSend($data)) {
      foreach ($this->storage as $store) {
        try {
          $store->write($data, $this);
        }
        catch (\Throwable $e) {
          // Capture any logger storage exceptions, but also allow other storage
          // plugins to still get called even though one failed.
          $logger = $this->getLogger('stenographer');
          Error::logException($logger, $e);
        }
      }
    }
  }

}
