<?php

namespace Drupal\stenographer\Trigger;

use Drupal\Core\Form\BaseFormIdInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\RecorderInterface;
use Drupal\stenographer\RecorderManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Helper function for modifying and attaching stenographer tracking to forms.
 */
class FormHandler {

  use LoggerChannelTrait;

  /**
   * Create a new instance of the FormEvents helper.
   *
   * @param \Drupal\stenographer\Trigger\TriggerBuilderInterface $triggers
   *   Trigger builder which creates the trigger definitions and mapping for
   *   the triggers which this handler executes.
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   The recorder strategy manager.
   */
  public function __construct(
    #[Autowire(service: 'stenographer.form_triggers')]
    public readonly TriggerBuilderInterface $triggers,
    protected RecorderManagerInterface $recorderManager,
  ) {}

  /**
   * Drupal form alter method to add tracker validation listeners to forms.
   *
   * @param string $formId
   *   The ID of the form being altered. This will be checked against the
   *   recorder settings for matches to the form.
   * @param array $form
   *   References to the form element structure to alter.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form build, state and user input for the form.
   */
  public function alterForm(string $formId, array &$form, FormStateInterface $formState): void {
    $formObj = $formState->getFormObject();

    $triggers[] = $formId;
    if ($formObj instanceof BaseFormIdInterface) {
      $triggers[] = $formObj->getBaseFormId();
    }

    $loggers = [];
    $validators = [];
    foreach ($triggers as $trigger) {
      $loggers += $this->recorderManager->getByTrigger('form', $trigger . ':submit');
      $validators += $this->recorderManager->getByTrigger('form', $trigger . ':validate');
    }

    if ($validators) {
      $form['#validate'][] = static::formValidate(...);
      $formState->setTemporaryValue('stenographer_validators', $this->recorderManager->getRecorders($validators));
    }
    if ($loggers) {
      $form['#submit'][] = static::formSubmit(...);
      $formState->setTemporaryValue('stenographer_loggers', $this->recorderManager->getRecorders($loggers));
    }
  }

  /**
   * Form validation handler to record to stenographer failed validation events.
   *
   * @param array $form
   *   Reference to the form structure and elements.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form build, state and user input for the form.
   */
  public static function formValidate(array &$form, FormStateInterface $formState): void {
    /** @var \Drupal\stenographer\RecorderInterface[] $recorders */
    $recorders = $formState->getTemporaryValue('stenographer_validators');

    if ($formState->hasAnyErrors() && $recorders) {
      $formId = $formState->getBuildInfo()['form_id'];

      foreach ($recorders as $recorder) {
        try {
          $event = static::getEventId($recorder, $formId, 'validate_fail');
          $recorder->logEvent('form', $event, ['form_state' => $formState]);
        }
        catch (\Throwable $e) {
          $logger = \Drupal::logger("stenographer:form:{$formId}");
          Error::logException($logger, $e);
        }
      }
    }
  }

  /**
   * Form submit handler to record events for successful the form submission.
   *
   * @param array $form
   *   Reference to the form structure and elements.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form build, state and user input for the form.
   */
  public static function formSubmit(array &$form, FormStateInterface $formState): void {
    $recorders = $formState->getTemporaryValue('stenographer_loggers') ?? [];
    $formId = $formState->getBuildInfo()['form_id'];
    $data['form_state'] = $formState;

    foreach ($recorders as $recorder) {
      try {
        $event = static::getEventId($recorder, $formId, 'validate');
        $recorder->logEvent('form', $event, $data);
      }
      catch (\Throwable $e) {
        $logger = \Drupal::logger("stenographer:form:{$formId}");
        Error::logException($logger, $e);
      }
    }
  }

  /**
   * Get the event ID or action ID to use when sending this event.
   *
   * @param \Drupal\stenographer\RecorderInterface $recorder
   *   The recorder instance the event is being generated for.
   * @param string $formId
   *   The form identifier.
   * @param string $action
   *   The form action being performed.
   *
   * @return string
   *   A string to use as the event or action ID of the event log.
   */
  protected static function getEventId(RecorderInterface $recorder, string $formId, string $action): string {
    $defaultEvent = $formId . ':' . $action;
    $triggerInfo = $recorder->getTriggers('form')[$formId] ?? $defaultEvent;

    if (is_array($triggerInfo)) {
      return $triggerInfo['label'] ?? $defaultEvent;
    }

    return is_string($triggerInfo) ? $triggerInfo : $defaultEvent;
  }

}
