<?php

namespace Drupal\stenographer\Hook;

use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\RecorderManagerInterface;
use Drupal\stenographer\StenographerKillSwitch;

/**
 * Create new functor to use as callback for Stenographer hook callbacks.
 */
class HookHandler {

  use LoggerChannelTrait;

  /**
   * Create a new instance of the logger hook callback.
   *
   * @param string $hook
   *   Name of the hook this instance is a trigger for.
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   The logger manager instance.
   * @param \Drupal\stenographer\StenographerKillSwitch $killSwitch
   *   The stenographer kill switch to halt recorder triggers.
   */
  public function __construct(
    public readonly string $hook,
    protected RecorderManagerInterface $recorderManager,
    protected StenographerKillSwitch $killSwitch,
  ) {}

  /**
   * Sends the logging events for registered loggers of this Drupal hook.
   */
  public function __invoke(...$args): void {
    if ($this->killSwitch->isHalted()) {
      return;
    }

    foreach ($this->recorderManager->getByTrigger('hook', $this->hook) as $loggerId) {
      try {
        $recorder = $this->recorderManager->getInstance($loggerId);
        $eventId = $recorder->getTriggers('hook')[$this->hook] ?? $this->hook;
        $recorder->logEvent('hook', $eventId, $args);
      }
      catch (\Throwable $e) {
        $loggerChannel = $this->getLogger("stenographer:hook:{$this->hook}");
        Error::logException($loggerChannel, $e);
      }
    }
  }

}
