<?php

namespace Drupal\stenographer\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Config\Entity\ConfigEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaInterface;
use Drupal\stenographer\Event\EntityAdapterMapAlter;
use Drupal\stenographer\Event\StenographerEvents;
use Drupal\stenographer\Plugin\Stenographer\Adapter\ConfigEntityAdapter;
use Drupal\stenographer\Plugin\Stenographer\Adapter\EntityAdapter;
use Drupal\stenographer\Plugin\Stenographer\Adapter\MediaEntityAdapter;
use Drupal\stenographer\Plugin\Stenographer\Adapter\UserEntityAdapter;
use Drupal\stenographer\Plugin\Stenographer\Adapter\UserRoleEntityAdapter;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Provides an entity data adapter for all content entity types.
 */
class EntityAdapterDeriver extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * Constructs new EntityDataAdapterDeriver.
   *
   * @param string $basePluginId
   *   The original plugin ID for which the derived definitions come from.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    protected string $basePluginId,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $base_plugin_id,
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($basePluginDef): array {
    if (!is_array($basePluginDef)) {
      $basePluginDef = [];
    }

    // Allow other modules to update the entity adapter class mapping.
    $adapterMap = [
      'media' => MediaEntityAdapter::class,
      'user' => UserEntityAdapter::class,
      'user_role' => UserRoleEntityAdapter::class,
    ];
    $this->eventDispatcher->dispatch(new EntityAdapterMapAlter($adapterMap), StenographerEvents::ENTITY_ADAPTER_MAP);

    $this->derivatives = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entityType) {
      $contextDef = EntityContextDefinition::fromEntityType($entityType);
      $contextDef->setLabel($entityType->getLabel());
      $contextDef->setRequired(FALSE);

      $entityTypeId = $entityType->id();
      $adapterClass = $adapterMap[$entityTypeId]
        ?? ($entityType instanceof ConfigEntityTypeInterface ? ConfigEntityAdapter::class : $basePluginDef['class']);

      if (EntityAdapter::class === $adapterClass && is_a($entityType->getClass(), MediaInterface::class, TRUE)) {
        $adapterClass = MediaEntityAdapter::class;
      }

      $this->derivatives[$entityTypeId] = [
        'class' => $adapterClass,
        'context_definitions' => [
          $entityType->id() => $contextDef,
        ],
      ] + $basePluginDef;
    }

    return $this->derivatives;
  }

}
