<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginTrait;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\stenographer\DataAdapterBase;
use Drupal\stenographer\TypedDataAdapterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Data adapter for retrieving data from an entity.
 */
class ConfigEntityAdapter extends DataAdapterBase implements TypedDataAdapterInterface, ContextAwarePluginInterface, ContainerFactoryPluginInterface {

  use ContextAwarePluginTrait;

  /**
   * The entity type ID for the adapter that this data expects.
   *
   * @var string
   */
  protected string $entityTypeId;

  /**
   * List of config entity property definitions.
   *
   * @var \Drupal\Core\TypedData\DataDefinitionInterface[]|null
   */
  protected ?array $properties = NULL;

  /**
   * Create a new instance of the current user data adapter.
   *
   * @param array $config
   *   The plugin configuration values.
   * @param string $pluginId
   *   The unique ID of the plugin.
   * @param mixed $definition
   *   The plugin definition from the plugin manager discovery.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $config,
    string $pluginId,
    $definition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config, $pluginId, $definition);

    // Capture the entity type ID for the plugin.
    [, $this->entityTypeId] = explode(':', $pluginId, 2);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * Fetches the entity to use as the data source for this adapter.
   *
   * @param array<string,mixed> $data
   *   Event context data to use when trying to extract entity values from.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The entity being worked on by this data adaptor.
   */
  protected function getEntity(array $data): EntityInterface {
    $entityKey = $this->configuration['entity_key'] ?? $this->entityTypeId;
    $entity = $data[$entityKey] ?? $data['entity'] ?? $this->getContextValue($entityKey);

    if (is_array($entity) && $entity['entity']) {
      $entity = $entity['entity'];
    }

    if ($entity instanceof EntityInterface && $entity->getEntityTypeId() === $this->entityTypeId) {
      return $entity;
    }

    $err = sprintf('Invalid context value. Expected entity of type %s but got %s', $this->entityTypeId, $entity->getEntityTypeId());
    throw new ContextException($err);
  }

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $properties = [];

    $entity = $this->getEntity($data);
    $entityType = $entity->getEntityType();

    foreach ($entity->getTypedData() as $key => $definition) {
      $properties[$key] = $definition;
    }

    $properties['@entity_type'] = DataDefinition::create('string')->setLabel('Entity type');
    $properties['@id'] = DataDefinition::create('string')->setLabel('Id');

    if ($entityType->getKey('label')) {
      $properties['@label'] = DataDefinition::create('string')->setLabel('Label');
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    $entity = $this->getEntity($data);
    /** @var \Drupal\Core\Config\Entity\ConfigEntityTypeInterface $entityType */
    $entityType = $entity->getEntityType();

    return match($name) {
      '@entity_type' => TRUE,
      '@id' => TRUE,
      '@label' => $entityType->hasKey('label'),
      default => (bool) ($this->propertyDefinitions($data)[$name] ?? FALSE),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    /** @var \Drupal\Core\Config\Entity\ConfigEntityInterface $entity */
    $entity = $this->getEntity($data);

    return match ($name) {
      '@entity_type' => $entity->getEntityTypeId(),
      '@id' => $entity->id(),
      '@label' => $entity->label(),
      default => $entity->get($name),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function applyData(array &$data, array $src, array $properties): void {
    parent::applyData($data, $src, $properties);

    if ($this->configuration['includeType'] ?? TRUE) {
      $entity = $this->getEntity($src);
      $data['targetType'] = $this->entityTypeId;
      $data['targetId'] = $entity->id();
    }
  }

}
