<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\stenographer\Attribute\StenographerAdapter;
use Drupal\stenographer\DataAdapterBase;
use Drupal\stenographer\TypedDataAdapterInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Exception\SessionNotFoundException;
use Symfony\Component\HttpFoundation\Request;

/**
 * Data adapter for exposing data from the current active user.
 */
#[StenographerAdapter('current_user')]
class CurrentUserAdapter extends DataAdapterBase implements TypedDataAdapterInterface, ContainerFactoryPluginInterface {

  /**
   * Entity type definition from the EntityTypeManager::getDefinition().
   *
   * @var \Drupal\Core\Entity\ContentEntityTypeInterface|null
   */
  protected ?ContentEntityTypeInterface $entityDef = NULL;

  /**
   * Currently active user account loaded. Can contain anonymous uer.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $currentUser;

  /**
   * Create a new instance of the current user data adapter.
   *
   * @param array $config
   *   The plugin configuration values.
   * @param string $plugin_id
   *   The unique ID of the plugin.
   * @param mixed $plugin_definition
   *   The plugin definition from the plugin manager discovery.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request being processed.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   Account proxy of the current user.
   * @param \Drupal\Core\Session\SessionManagerInterface $sessionManager
   *   The user session manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service for getting entity definitions and
   *   entity handler classes.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $fieldManager
   *   Manager service for getting bundle field definitions for entities.
   */
  public function __construct(
    array $config,
    string $plugin_id,
    $plugin_definition,
    protected Request $request,
    AccountProxyInterface $account,
    protected SessionManagerInterface $sessionManager,
    EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $fieldManager,
  ) {
    parent::__construct($config, $plugin_id, $plugin_definition);

    /** @var \Drupal\Core\Entity\ContentEntityType $entityType */
    $entityType = $entityTypeManager->getDefinition('user');
    $this->entityDef = $entityType;
    $this->currentUser = $entityTypeManager->getStorage('user')->load($account->id());
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('current_user'),
      $container->get('session_manager'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $bundle = $this->currentUser->bundle();

    $properties = [];
    foreach ($this->fieldManager->getFieldDefinitions('user', $bundle) as $fieldDef) {
      $storageDef = $fieldDef->getFieldStorageDefinition();
      /** @var class-string<\Drupal\Core\Field\FieldItemBase> $fieldClass */
      $fieldClass = $fieldDef->getClass();
      $fieldPropName = ($fieldClass . '::mainPropertyName')();
      $properties[$fieldDef->getName()] = $fieldClass::propertyDefinitions($storageDef)[$fieldPropName];
    }

    $properties['session'] = DataDefinition::create('string')->setLabel('Hashed user session');

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    return 'session' === $name || $this->currentUser->hasField($name);
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    if ('session' === $name) {
      $hashed = NULL;

      if ($this->currentUser->isAuthenticated()) {
        try {
          $session = $this->request->getSession();
          $sessionKey = 'stenographer_session_hash';

          if (!($hashed = $session->get($sessionKey, NULL))) {
            $hashed = Crypt::hashBase64($this->sessionManager->getId());
            $session->set($sessionKey, $hashed);
          }
        }
        catch (SessionNotFoundException $e) {
          // Unable to fetch the Symfony, fallback to building on they fly.
          $hashed = Crypt::hashBase64($this->sessionManager->getId());
        }
      }
      return $hashed;
    }

    if ($this->hasProperty($name, $data) && !$this->currentUser->{$name}->isEmpty()) {
      $field = $this->currentUser->{$name}->first();
      $mainProperty = (get_class($field) . '::mainPropertyName')();

      return !empty($mainProperty) ? $field->{$mainProperty} : NULL;
    }

    return NULL;
  }

}
