<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem;
use Drupal\media\MediaInterface;

/**
 * Data adapter for retrieving data from a media entity.
 *
 * Adds "@file" meta field to provide underlying media file source properties
 * if the media type uses a file entity as its source.
 *
 * @todo implement "@source" to provide other to provide media source properties
 * and values for media types that aren't using a file entity.
 */
class MediaEntityAdapter extends EntityAdapter {

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $entity = $this->getEntity($data);
    $entityType = $entity->getEntityType();
    $bundle = $entity->bundle();

    if (!isset(self::$entityProperties[$entityType->id()][$bundle])) {
      $properties = parent::propertyDefinitions($data);

      if ($entity instanceof MediaInterface && ($fieldDef = $this->getMediaSourceDefinition($entity))) {
        $properties = &self::$entityProperties[$entityType->id()][$bundle];
        $itemClass = $fieldDef->getItemDefinition()->getClass();

        if ('file' === $fieldDef->getSetting('target_type') && is_a($itemClass, EntityReferenceItem::class, TRUE)) {
          $fileType = $this->entityTypeManager->getDefinition('file');

          foreach ($this->fieldManager->getBaseFieldDefinitions($fileType->id()) as $definition) {
            $storageDef = $definition->getFieldStorageDefinition();
            $fieldPropName = $storageDef->getMainPropertyName();
            $properties['@file' . $fieldDef->getName()] = $storageDef->getPropertyDefinition($fieldPropName);
          }
        }
      }
    }

    return self::$entityProperties[$entityType->id()][$bundle];
  }

  /**
   * {@inheritdoc}
   */
  public function hasProperty(string $name, array $data): bool {
    @[$field, $property] = explode('.', $name, 2);
    $entity = $this->getEntity($data);

    if ('@file' === $field && $entity instanceof MediaInterface) {
      if ($fieldDef = $this->getMediaSourceDefinition($entity)) {
        $itemClass = $fieldDef->getItemDefinition()->getClass();

        if ('file' === $fieldDef->getSetting('target_type') && is_a($itemClass, EntityReferenceItem::class, TRUE)) {
          return isset($this->getFileFields()[$property]);
        }
      }

      return FALSE;
    }

    return parent::hasProperty($name, $data);
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    @[$field, $property] = explode('.', $name, 2);
    $entity = $this->getEntity($data);

    if ('@file' === $field && $entity instanceof MediaInterface) {
      $fieldDef = $this->getMediaSourceDefinition($entity);
      $itemClass = $fieldDef?->getItemDefinition()->getClass();

      if ($fieldDef && 'file' === $fieldDef->getSetting('target_type') && is_a($itemClass, EntityReferenceItem::class, TRUE)) {
        try {
          @[$fieldId, $fileProperty] = explode('.', $property, 2);
          /** @var \Drupal\file\FileInterface $file */
          $file = $entity->get($fieldDef->getName())->entity;
          $fileField = $file->get($fieldId);

          if ($fileField && !$fileField->isEmpty()) {
            $valueProperty = $fileProperty ?: (get_class($fileField->first()) . '::mainPropertyName')();

            if (!empty($valueProperty)) {
              if ($fileField->count() > 1) {
                $values = [];
                foreach ($fileField as $item) {
                  $values[] = $item->{$valueProperty};
                }

                return $values;
              }

              return $fileField->first()?->{$valueProperty};
            }
          }
        }
        catch (\InvalidArgumentException $e) {
          // Requested field doesn't exist for files.
        }
      }

      return NULL;
    }

    return parent::get($name, $data);
  }

  /**
   * Get the field definition of the media source.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity to get the field definition for if it exists.
   *
   * @return \Drupal\Core\Field\FieldDefinitionInterface|null
   *   The field definition of the provided media source if defined.
   */
  protected function getMediaSourceDefinition(MediaInterface $media): ?FieldDefinitionInterface {
    $bundleId = $media->getEntityType()->getKey('bundle');
    /** @var \Drupal\media\MediaTypeInterface $mediaType */
    $mediaType = $media->get($bundleId)->entity;
    return $media->getSource()->getSourceFieldDefinition($mediaType);
  }

  /**
   * Get the base file entity field definitions.
   *
   * @return \Drupal\Core\Field\FieldDefinitionInterface[]
   *   Field definitions for all file entity base field definitions.
   */
  protected function getFileFields(): array {
    static $fileFields = NULL;

    if (NULL === $fileFields) {
      $fileType = $this->entityTypeManager->getDefinition('file');
      $fileFields = $this->fieldManager->getBaseFieldDefinitions($fileType->id());
    }

    return $fileFields;
  }

}
