<?php

namespace Drupal\stenographer\Plugin\Stenographer\Adapter;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\ListDataDefinition;
use Drupal\stenographer\TypedDataAdapterInterface;
use Drupal\user\RoleInterface;

/**
 * Data adapter for retrieving data from an entity.
 */
class UserRoleEntityAdapter extends ConfigEntityAdapter implements TypedDataAdapterInterface {

  /**
   * {@inheritdoc}
   */
  public function propertyDefinitions(array $data): array {
    $properties = parent::propertyDefinitions($data);

    $properties['@added_permissions'] = ListDataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Added permissions'));

    $properties['@removed_permissions'] = ListDataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Removed permissions'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $name, array $data): mixed {
    /** @var \Drupal\user\RoleInterface $role */
    $role = $this->getEntity($data);

    return match ($name) {
      '@added_permissions' => $this->addedPermissions($role),
      '@removed_permissions' => $this->removedPermissions($role),
      default => parent::get($name, $data),
    };
  }

  /**
   * Determine if any permissions have been added to a role.
   *
   * @param \Drupal\user\RoleInterface $role
   *   The user role permissions to check for added permissions.
   *
   * @return array
   *   A list of permissions which have been added.
   */
  public function addedPermissions(RoleInterface $role): array {
    $original = $role->original ?? NULL;

    return $original
      ? array_diff($role->getPermissions(), $original->getPermissions())
      : $role->getPermissions();
  }

  /**
   * Get a list of permissions which have been removed from this role.
   *
   * @param \Drupal\user\RoleInterface $role
   *   The user role permissions to check for removed permissions.
   *
   * @return array
   *   A list of permissions which have been removed.
   */
  public function removedPermissions(RoleInterface $role): array {
    $original = $role->original ?? NULL;

    return $original
      ? array_diff($original->getPermissions(), $role->getPermissions())
      : [];
  }

}
