<?php

namespace Drupal\stenographer\Plugin\Stenographer\Storage;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\Attribute\StenographerStore;
use Drupal\stenographer\RecorderInterface;
use Drupal\stenographer\StoreInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Stenographer recorder storage using a database table.
 *
 * Assumes that the tables are defined by a module schema and already installed.
 * Configurations are required to specify the table and the module which
 * provides it.
 *
 * @todo Determine table schema without needing the module which defined it.
 */
#[StenographerStore('db_table')]
class DatabaseStore extends PluginBase implements StoreInterface, ContainerFactoryPluginInterface {

  use LoggerChannelTrait;

  /**
   * Table name of the table which this store logs to.
   *
   * @var string|null
   */
  protected ?string $logTable = NULL;

  /**
   * Table schema definition for target logging table.
   *
   * @var array
   */
  protected array $schema = [];

  /**
   * Create a new instance of a database storage logger.
   *
   * @param array $config
   *   Configuration array containing database table to log to.
   * @param string $plugin_id
   *   The unique plugin ID.
   * @param mixed $plugin_definition
   *   Definition of the plugin as discovered by the plugin manager.
   * @param \Drupal\Core\Database\Connection $db
   *   Open database connection for creating queries.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    array $config,
    string $plugin_id,
    $plugin_definition,
    protected Connection $db,
    protected ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($config, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $config, $plugin_id, $plugin_definition) {
    return new static(
      $config,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = [];
    $dependencies['module'][] = 'stenographer';

    // Ensure that the module providing the table schema and install is included
    // in the required modules for this configuration.
    if (!empty($this->configuration['table_provider'])) {
      $dependencies['module'][] = $this->configuration['table_provider'];
    }

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration;

    if (isset($this->logTable) && $configuration['table'] !== $this->logTable) {
      $this->refreshTable();
    }
  }

  /**
   * Refresh the information about the database table.
   */
  protected function refreshTable(): void {
    if (empty($this->configuration['table'])) {
      throw new \InvalidArgumentException('The database storage configuration is missing a table name to log to.');
    }

    if (empty($this->logTable)) {
      $provider = $this->configuration['table_provider'];

      $this->logTable = $this->db->escapeTable($this->configuration['table']);

      if ($this->moduleHandler->loadInclude($provider, 'install')) {
        $this->schema = $this->moduleHandler->invoke($provider, 'schema')[$this->logTable] ?? [];
      }
      else {
        $this->schema = [];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function write(array $data, RecorderInterface $recorder): void {
    try {
      if (!isset($this->logTable)) {
        $this->refreshTable();
      }

      if (!empty($this->configuration['map'])) {
        foreach ($this->configuration['map'] as $key => $source) {
          $data[$key] = $data[$source];
        }
      }

      if (!empty($data)) {
        $this->db->insert($this->logTable)
          ->fields(array_intersect_key($data, $this->schema['fields']))
          ->execute();
      }
    }
    catch (\Throwable $e) {
      Error::logException($this->getLogger('stenographer'), $e);
    }
  }

}
