<?php

namespace Drupal\stenographer;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\stenographer\Hook\HookHandler;
use Symfony\Component\DependencyInjection\Exception\ParameterNotFoundException;

/**
 * Module service provider to adjust hook implementation definitions.
 */
class StenographerServiceProvider extends ServiceProviderBase {

  /**
   * Add hook definitions to the "hook_implementations_map" container parameter.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The dependency injection container builder.
   */
  public function alter(ContainerBuilder $container): void {
    try {
      $loaded = FALSE;
      $implementations = $container->getParameter('hook_implementations_map');
      $hooks = function_exists('apcu_fetch') ? \apcu_fetch('stenographer:hooks', $loaded) : [];

      if (!$loaded) {
        $hooks = [];
        $dirs = [];

        // Find Stenographer recorders that trigger on hooks, and add them to
        // the list of hooks that need to be called by a recorder.
        foreach ($container->getParameter('container.modules') as $module => $info) {
          if ('module' === $info['type'] && !empty($info['pathname'])) {
            $dirs[$module] = \dirname($info['pathname']);
          }
        }

        $discovery = RecorderManager::createDiscovery($dirs);
        foreach ($discovery->findAll() as $fileDef) {
          foreach ($fileDef as $loggerDef) {
            $keys = \array_keys($loggerDef['trigger']['hook'] ?? []);
            $hooks += \array_combine($keys, $keys);
          }
        }

        \apcu_store('stenographer:hooks', \array_unique($hooks));
      }

      // Add our hooks to the implementation map.
      foreach ($hooks as $hook) {
        $implementations[$hook][HookHandler::class]['__invoke'] = 'stenographer';
      }
      $container->setParameter('hook_implementations_map', $implementations);
    }
    catch (ParameterNotFoundException) {
      // The 'hook_implementations_map' container parameter is not defined
      // before Drupal 11 and will throw this exception when not available.
    }
  }

}
