<?php

namespace Drupal\stenographer\Trigger;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\RecorderManagerInterface;
use Drupal\stenographer\StenographerKillSwitch;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Service / utility for consistent logging of entity events.
 */
class EntityHandler {

  use LoggerChannelTrait;

  /**
   * Creates a new instance of the EntityLogger helper.
   *
   * @param \Drupal\stenographer\Trigger\TriggerBuilderInterface $triggers
   *   Trigger builder which creates the trigger definitions and mapping for
   *   the triggers which this handler executes.
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   The recorder strategy manager.
   * @param \Drupal\stenographer\StenographerKillSwitch $killSwitch
   *   The stenographer kill switch to halt recorder triggers.
   */
  public function __construct(
    #[Autowire(service: 'stenographer.entity_triggers')]
    public readonly TriggerBuilderInterface $triggers,
    protected RecorderManagerInterface $recorderManager,
    protected StenographerKillSwitch $killSwitch,
  ) {}

  /**
   * Record the entity events based on the entity action.
   *
   * @param string $action
   *   The entity operation and actions that is being logged.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The target entity for the event.
   * @param array $context
   *   The context of the event trigging the log. The event can contain
   *   the entity display and view_mode if the event being captured is an view
   *   action for the entity.
   */
  public function logEvents(string $action, EntityInterface $entity, array $context = []): void {
    if ($this->killSwitch->isHalted()) {
      return;
    }

    $entityType = $entity->getEntityType();
    $entityTypeId = $entityType->id();
    $bundle = $entityType->hasKey('bundle') ? $entity->bundle() : NULL;

    $data['entity'] = [
      'entity' => $entity,
      'context' => $context,
    ];
    $data[$entityTypeId] = $data['entity'];
    $eventType = "{$entityTypeId}:{$action}";

    if ('view' === $action && !empty($context['view_mode'])) {
      $eventType .= ':' . $context['view_mode'];
    }

    $ids = [];
    foreach ($this->getTriggers($action, $entityTypeId) as $trigger) {
      $ids += $this->recorderManager->getByTrigger('entity', $trigger);
    }

    foreach ($ids as $rid) {
      try {
        $recorder = $this->recorderManager->getInstance($rid);
        $triggerDef = $recorder->getTriggers('entity')[$entityTypeId] ?? [];
        $allowedBundles = $triggerDef['bundles'] ?? NULL;

        // Test if trigger is restricted to specific entity bundles.
        if ($bundle && $allowedBundles && !in_array($bundle, $allowedBundles)) {
          continue;
        }

        $recorder->logEvent('entity', $triggerDef['action'] ?? $eventType, $data);
      }
      catch (\Throwable $e) {
        $logger = $this->getLogger("stenographer:entity:{$entityTypeId}:{$action}");
        Error::logException($logger, $e);
      }
    }
  }

  /**
   * Get the event triggers that are mapped from the entity actions.
   *
   * @param string $action
   *   The entity operation / action which is triggering the event.
   * @param string $entityTypeId
   *   The entity type ID for the entity which is triggering this event.
   *
   * @return string[]
   *   A list of entity triggers to create log events for.
   */
  protected function getTriggers(string $action, string $entityTypeId): array {
    return match($action) {
      'create', 'insert' => [
        $entityTypeId . ':save',
        $entityTypeId . ':create',
      ],
      'view' => [
        $entityTypeId . ':view',
      ],
      'update', 'edit' => [
        $entityTypeId . ':save',
        $entityTypeId . ':update',
      ],
      'delete', 'remove' => [
        $entityTypeId . ':delete',
      ],
      default => [],
    };
  }

}
