<?php

namespace Drupal\stenographer\Trigger;

use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\stenographer\RecorderManagerInterface;
use Drupal\stenographer\StenographerKillSwitch;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Helper function for modifying and attaching tracking events to exceptions.
 */
class ExceptionHandler {

  use LoggerChannelTrait;

  /**
   * Create a new instance of the FormEvents helper.
   *
   * @param \Drupal\stenographer\Trigger\TriggerBuilderInterface $triggers
   *   Trigger builder which creates the trigger definitions and mapping for
   *   the triggers which this handler executes.
   * @param \Drupal\stenographer\RecorderManagerInterface $recorderManager
   *   The recorder strategy manager.
   * @param \Drupal\stenographer\StenographerKillSwitch $killSwitch
   *   The stenographer kill switch to halt recorder triggers.
   */
  public function __construct(
    #[Autowire(service: 'stenographer.exception_triggers')]
    public readonly TriggerBuilderInterface $triggers,
    protected RecorderManagerInterface $recorderManager,
    protected StenographerKillSwitch $killSwitch,
  ) {}

  /**
   * Log exceptions with a stenographer recorder.
   *
   * @param \Throwable $exception
   *   Log an exception through the stenographer recorders.
   */
  public function logException(\Throwable $exception): void {
    if ($this->killSwitch->isHalted()) {
      return;
    }

    $exceptType = NULL;
    if ($exception instanceof HttpException) {
      $exceptType = match (TRUE) {
        $exception instanceof AccessDeniedHttpException => 'access',
        $exception instanceof NotFoundHttpException => 'not_found',
        default => NULL,
      };
    }

    if (!$exceptType) {
      return;
    }

    $triggerType = $this->triggers->getTriggerType();
    foreach ($this->recorderManager->getByTrigger('exception', $exceptType) as $id) {
      try {
        $recorder = $this->recorderManager->getInstance($id);
        $eventId = $recorder->getTriggers($triggerType)[$exceptType] ?? "exception: {$exceptType}";
        $recorder->logEvent('exception', $eventId, ['exception' => $exception]);
      }
      catch (\Throwable $e) {
        Error::logException($this->getLogger('stenographer'), $e);
      }
    }
  }

}
