<?php

namespace Drupal\stenographer\Trigger;

use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Contracts\Service\ServiceCollectionInterface;

/**
 * Service to collect the stenographer trigger handlers.
 */
class TriggerCollection implements ServiceCollectionInterface {

  /**
   * List of registered trigger builders.
   *
   * @var \Drupal\stenographer\Trigger\TriggerBuilderInterface[]
   */
  protected array $builders = [];

  /**
   * {@inheritdoc}
   */
  public function has(string $id): bool {
    return !empty($this->builders[$id]);
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $id): TriggerBuilderInterface {
    return $this->builders[$id] ?? throw new ServiceNotFoundException(
      id: $id,
      alternatives: []
    );
  }

  /**
   * {@inheritdoc}
   */
  public function count(): int {
    return count($this->builders);
  }

  /**
   * Get an iterator for traversing all registered trigger builders.
   *
   * @return \Traversable<string,\Drupal\stenographer\Trigger\TriggerBuilderInterface>
   *   An iterator to traverse the registered trigger builder services.
   */
  public function getIterator(): \Traversable {
    return new \ArrayIterator($this->builders);
  }

  /**
   * {@inheritdoc}
   */
  public function getProvidedServices(): array {
    $services = [];
    foreach ($this->getBuilders() as $key => $handler) {
      $services[$key] = \get_class($handler);
    }

    return $services;
  }

  /**
   * Get array of all the registered trigger handler services.
   *
   * @return \Drupal\stenographer\Trigger\TriggerBuilderInterface[]
   *   A list of the trigger builders.
   */
  public function getBuilders(): array {
    return $this->builders;
  }

  /**
   * Add a trigger handler to the collection.
   *
   * @param \Drupal\stenographer\Trigger\TriggerBuilderInterface $builder
   *   The trigger builder to add to the collection.
   * @param string $id
   *   The trigger handler identifier.
   */
  public function addBuilder(TriggerBuilderInterface $builder, string $id): void {
    $this->builders[$id] = $builder;
  }

}
