<?php

namespace Drupal\sticky_audio_player\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;

/**
 * Plugin implementation of the 'sticky_audio_player' formatter.
 *
 * @FieldFormatter(
 *   id = "sticky_audio_player",
 *   label = @Translation("Sticky Audio Player"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class StickyAudioPlayerFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'button_text' => 'Listen to Article',
      'button_icon' => '🎧',
      'autostart' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button text'),
      '#default_value' => $this->getSetting('button_text'),
      '#description' => $this->t('The text displayed on the button.'),
    ];

    $elements['button_icon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button icon'),
      '#default_value' => $this->getSetting('button_icon'),
      '#description' => $this->t('Icon or emoji to display before the button text.'),
    ];

    $elements['autostart'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autoplay on button click'),
      '#default_value' => $this->getSetting('autostart'),
      '#description' => $this->t('Automatically start playing the audio when the button is clicked.'),
    ];

    // Add information about global navigation persistence settings.
    $elements['navigation_info'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Navigation persistence settings are configured globally. <a href=":url">Configure navigation persistence settings</a>.', [
        ':url' => Url::fromRoute('sticky_audio_player.settings')->toString(),
      ]) . '</p>',
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Button text: @text', ['@text' => $this->getSetting('button_text')]);
    $autostart = $this->getSetting('autostart') ? $this->t('Yes') : $this->t('No');
    $summary[] = $this->t('Autoplay: @autostart', ['@autostart' => $autostart]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $settings = $this->getSettings();
    $entity = $items->getEntity();

    // Attach library once for all items.
    $elements['#attached']['library'][] = 'sticky_audio_player/player';

    foreach ($items as $delta => $item) {
      if (!$item->entity) {
        continue;
      }

      $media = $item->entity;
      $file = NULL;
      
      // Handle both Media entities and direct File entities.
      if ($media instanceof Media) {
        try {
          $source_config = $media->getSource()->getConfiguration();
          if (isset($source_config['source_field']) && $media->hasField($source_config['source_field'])) {
            $file_field = $media->get($source_config['source_field']);
            if (!$file_field->isEmpty() && $file_field->entity) {
              $file = $file_field->entity;
            }
          }
        }
        catch (\Exception $e) {
          // Skip if we can't get the source field.
          continue;
        }
      }
      elseif ($media instanceof File) {
        $file = $media;
      }

      // Only process audio files.
      if (!$file || !$file instanceof File) {
        continue;
      }

      $mime_type = $file->getMimeType();
      $audio_mime_types = [
        'audio/mpeg',
        'audio/mp3',
        'audio/mpeg3',
        'audio/x-mpeg-3',
        'audio/wav',
        'audio/ogg',
        'audio/webm',
        'audio/x-wav',
      ];

      if (!in_array($mime_type, $audio_mime_types)) {
        continue;
      }

      // Get the file URL.
      $file_url = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
      
      // Generate a unique ID for this audio instance.
      $unique_id = 'audio-' . $media->id() . '-' . $delta;
      
      // Get the entity title (for display in player).
      $title = $entity && $entity->label() ? $entity->label() : $this->t('Audio');
      
      // Build the button render array - ensure it's a proper render array.
      $elements[$delta] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['sticky-audio-player-button-wrapper'],
        ],
        'button' => [
          '#type' => 'html_tag',
          '#tag' => 'button',
          '#attributes' => [
            'class' => ['listen-to-article-btn'],
            'type' => 'button',
            'data-audio-id' => $unique_id,
            'data-audio-url' => $file_url,
            'data-article-title' => $title,
            'aria-label' => $this->t('Listen to @title', ['@title' => $title]),
          ],
          '#value' => $settings['button_icon'] . ' ' . $settings['button_text'],
        ],
      ];

      // Add settings for this audio instance.
      $elements['#attached']['drupalSettings']['stickyAudioPlayer'][$unique_id] = [
        'url' => $file_url,
        'title' => $title,
        'autostart' => !empty($settings['autostart']),
      ];
    }

    return $elements;
  }

}

