/**
 * @file
 * Sticky audio player JavaScript functionality.
 */

(function (Drupal, drupalSettings, htmx) {
  'use strict';

  /**
   * StickyAudioPlayer class - Singleton pattern for audio playback.
   */
  class StickyAudioPlayer {
    constructor() {
      this.audio = null;
      this.currentAudioId = null;
      this.isPlaying = false;
      this.currentTime = 0;
      this.duration = 0;
      this.playbackSpeed = 1.0;
      this.playerElement = null;
      this.progressBar = null;
      this.currentTimeDisplay = null;
      this.durationDisplay = null;
      this.playPauseButton = null;
      this.speedButton = null;
      this.closeButton = null;
      this.titleElement = null;
      this.currentTitle = '';
      this.currentUrl = '';
      this.htmxInitialized = false;
      this.htmxClickHandlerAdded = false;
      this.navigationEnabled = false;
      this.contentSelector = 'main, .layout-content';
      this.excludeSelectors = 'a[target="_blank"], a[href^="http"], a[href^="//"], a[data-no-htmx], .toolbar a, .admin-toolbar a, form a, a[href^="javascript:"], a[href="#"]';
      this.stateRestored = false;
      this.userClosed = false;

      this.init();
    }

    /**
     * Initialize the player.
     */
    init() {
      // Ensure persistent audio element exists first.
      this.audio = document.getElementById('persistent-audio-player');
      if (!this.audio) {
        this.audio = document.createElement('audio');
        this.audio.id = 'persistent-audio-player';
        this.audio.setAttribute('data-initialized', 'false');
        document.body.appendChild(this.audio);
      }

      this.createPlayerUI();
      this.attachEventListeners();
      this.adjustForSidebar();
      
      // Listen for window resize and sidebar changes.
      window.addEventListener('resize', () => {
        this.adjustForSidebar();
      });

      // Watch for sidebar toggle (Gin theme uses body class changes).
      const observer = new MutationObserver(() => {
        this.adjustForSidebar();
      });
      observer.observe(document.body, {
        attributes: true,
        attributeFilter: ['class']
      });
    }

    /**
     * Create the player UI structure.
     */
    createPlayerUI() {
      // Check if player already exists (from previous page).
      let existingPlayer = document.getElementById('sticky-audio-player');
      if (existingPlayer) {
        this.playerElement = existingPlayer;
        // Restore references to existing elements.
        this.playPauseButton = this.playerElement.querySelector('.play-pause');
        this.progressBar = this.playerElement.querySelector('.progress-bar');
        this.currentTimeDisplay = this.playerElement.querySelector('.current-time');
        this.durationDisplay = this.playerElement.querySelector('.duration');
        this.speedButton = this.playerElement.querySelector('.speed-control');
        this.closeButton = this.playerElement.querySelector('.close-button');
        this.titleElement = this.playerElement.querySelector('.player-title');
        // Audio element should already exist from init().
        if (!this.audio) {
          this.audio = document.getElementById('persistent-audio-player');
        }
        return;
      }

      // Create main player container.
      this.playerElement = document.createElement('div');
      this.playerElement.id = 'sticky-audio-player';
      this.playerElement.className = 'sticky-audio-player';
      this.playerElement.setAttribute('role', 'region');
      this.playerElement.setAttribute('aria-label', Drupal.t('Audio player'));

      // Player content wrapper.
      const playerContent = document.createElement('div');
      playerContent.className = 'player-content';

      // Left section: Play/Pause button and info.
      const leftSection = document.createElement('div');
      leftSection.className = 'player-left';

      // Play/Pause button.
      this.playPauseButton = document.createElement('button');
      this.playPauseButton.className = 'player-control play-pause';
      this.playPauseButton.setAttribute('type', 'button');
      this.playPauseButton.setAttribute('aria-label', Drupal.t('Play'));
      this.playPauseButton.innerHTML = '<span class="icon">▶</span>';
      leftSection.appendChild(this.playPauseButton);

      // Title and time info.
      const infoSection = document.createElement('div');
      infoSection.className = 'player-info';

      this.titleElement = document.createElement('div');
      this.titleElement.className = 'player-title';
      infoSection.appendChild(this.titleElement);

      const timeInfo = document.createElement('div');
      timeInfo.className = 'player-time';
      this.currentTimeDisplay = document.createElement('span');
      this.currentTimeDisplay.className = 'current-time';
      this.currentTimeDisplay.textContent = '0:00';
      this.durationDisplay = document.createElement('span');
      this.durationDisplay.className = 'duration';
      this.durationDisplay.textContent = '0:00';
      timeInfo.appendChild(this.currentTimeDisplay);
      timeInfo.appendChild(document.createTextNode(' / '));
      timeInfo.appendChild(this.durationDisplay);
      infoSection.appendChild(timeInfo);

      leftSection.appendChild(infoSection);

      // Center section: Progress bar.
      const centerSection = document.createElement('div');
      centerSection.className = 'player-center';

      const progressWrapper = document.createElement('div');
      progressWrapper.className = 'progress-wrapper';

      this.progressBar = document.createElement('div');
      this.progressBar.className = 'progress-bar';
      this.progressBar.setAttribute('role', 'slider');
      this.progressBar.setAttribute('aria-label', Drupal.t('Audio progress'));
      this.progressBar.setAttribute('tabindex', '0');
      this.progressBar.setAttribute('aria-valuemin', '0');
      this.progressBar.setAttribute('aria-valuemax', '100');
      this.progressBar.setAttribute('aria-valuenow', '0');

      const progressFill = document.createElement('div');
      progressFill.className = 'progress-fill';
      this.progressBar.appendChild(progressFill);

      progressWrapper.appendChild(this.progressBar);
      centerSection.appendChild(progressWrapper);

      // Right section: Speed control and close button.
      const rightSection = document.createElement('div');
      rightSection.className = 'player-right';

      // Speed control.
      this.speedButton = document.createElement('button');
      this.speedButton.className = 'player-control speed-control';
      this.speedButton.setAttribute('type', 'button');
      this.speedButton.setAttribute('aria-label', Drupal.t('Playback speed'));
      this.speedButton.textContent = '1x';
      rightSection.appendChild(this.speedButton);

      // Close button.
      this.closeButton = document.createElement('button');
      this.closeButton.className = 'player-control close-button';
      this.closeButton.setAttribute('type', 'button');
      this.closeButton.setAttribute('aria-label', Drupal.t('Close player'));
      this.closeButton.innerHTML = '<span class="icon">×</span>';
      rightSection.appendChild(this.closeButton);

      // Assemble the player.
      playerContent.appendChild(leftSection);
      playerContent.appendChild(centerSection);
      playerContent.appendChild(rightSection);
      this.playerElement.appendChild(playerContent);

      // Append player to body.
      document.body.appendChild(this.playerElement);
    }

    /**
     * Attach event listeners.
     */
    attachEventListeners() {
      // Play/Pause button.
      this.playPauseButton.addEventListener('click', () => {
        this.togglePlay();
      });

      // Progress bar click.
      this.progressBar.addEventListener('click', (e) => {
        const rect = this.progressBar.getBoundingClientRect();
        const percent = (e.clientX - rect.left) / rect.width;
        this.seek(percent * this.duration);
      });

      // Progress bar keyboard navigation.
      this.progressBar.addEventListener('keydown', (e) => {
        if (e.key === 'ArrowLeft' || e.key === 'ArrowRight') {
          e.preventDefault();
          const delta = e.key === 'ArrowLeft' ? -5 : 5;
          const newTime = Math.max(0, Math.min(this.duration, this.currentTime + delta));
          this.seek(newTime);
        }
      });

      // Speed control.
      this.speedButton.addEventListener('click', () => {
        this.cycleSpeed();
      });

      // Close button.
      this.closeButton.addEventListener('click', () => {
        this.hide();
      });

      // Keyboard shortcuts.
      document.addEventListener('keydown', (e) => {
        if (this.playerElement.classList.contains('visible')) {
          if (e.code === 'Space' && e.target.tagName !== 'BUTTON' && e.target.tagName !== 'INPUT' && e.target.tagName !== 'TEXTAREA') {
            e.preventDefault();
            this.togglePlay();
          }
        }
      });
    }

    /**
     * Initialize audio element event listeners (only once).
     */
    initializeAudioListeners() {
      if (this.audio.getAttribute('data-initialized') === 'true') {
        return;
      }

      // Debounced state save for timeupdate.
      let saveTimeout;
      this.audio.addEventListener('timeupdate', () => {
        this.updateProgress();
        // Debounce state saving (every 1 second max).
        clearTimeout(saveTimeout);
        saveTimeout = setTimeout(() => {
          this.saveState();
        }, 1000);
      });

      this.audio.addEventListener('play', () => {
        this.isPlaying = true;
        this.updatePlayPauseButton();
        this.saveState();
      });

      this.audio.addEventListener('pause', () => {
        this.isPlaying = false;
        this.updatePlayPauseButton();
        this.saveState();
      });

      this.audio.addEventListener('loadedmetadata', () => {
        this.duration = this.audio.duration;
        if (this.durationDisplay) {
          this.durationDisplay.textContent = this.formatTime(this.duration);
        }
        if (this.progressBar) {
          this.progressBar.setAttribute('aria-valuemax', Math.floor(this.duration));
        }
      });

      this.audio.addEventListener('ended', () => {
        this.isPlaying = false;
        this.updatePlayPauseButton();
        this.clearState();
      });

      this.audio.addEventListener('error', () => {
        alert(Drupal.t('Error loading audio file.'));
        this.hide();
      });

      // Save state before page unload.
      window.addEventListener('pagehide', () => {
        this.saveState();
      });

      this.audio.setAttribute('data-initialized', 'true');
    }

    /**
     * Load audio file.
     */
    loadAudio(url, title, audioId, autostart = false) {
      // If same audio is already loaded, just show player.
      if (this.currentAudioId === audioId && this.audio && this.audio.src === url) {
        this.show();
        if (autostart && !this.isPlaying) {
          this.togglePlay();
        }
        return;
      }

      // Store current state.
      this.currentUrl = url;
      this.currentTitle = title;
      this.currentAudioId = audioId;

      // Ensure audio element exists.
      if (!this.audio) {
        this.audio = document.getElementById('persistent-audio-player');
        if (!this.audio) {
          this.audio = document.createElement('audio');
          this.audio.id = 'persistent-audio-player';
          document.body.appendChild(this.audio);
        }
      }

      // Initialize listeners if needed.
      this.initializeAudioListeners();

      // Load new audio.
      this.audio.src = url;
      this.audio.playbackRate = this.playbackSpeed;

      // Update title.
      if (this.titleElement) {
        this.titleElement.textContent = title;
      }

      // Show player.
      this.show();
      this.userClosed = false;

      // Autostart if enabled.
      if (autostart) {
        this.audio.play().catch(() => {});
      }
    }

    /**
     * Toggle play/pause.
     */
    togglePlay() {
      if (!this.audio) {
        return;
      }

      if (this.isPlaying) {
        this.audio.pause();
        this.isPlaying = false;
      } else {
        this.audio.play().catch((error) => {
          console.error('Error playing audio:', error);
        });
        this.isPlaying = true;
      }

      this.updatePlayPauseButton();
    }

    /**
     * Update play/pause button state.
     */
    updatePlayPauseButton() {
      if (this.isPlaying) {
        this.playPauseButton.innerHTML = '<span class="icon">⏸</span>';
        this.playPauseButton.setAttribute('aria-label', Drupal.t('Pause'));
      } else {
        this.playPauseButton.innerHTML = '<span class="icon">▶</span>';
        this.playPauseButton.setAttribute('aria-label', Drupal.t('Play'));
      }
    }

    /**
     * Seek to specific time.
     */
    seek(time) {
      if (!this.audio) {
        return;
      }

      this.audio.currentTime = time;
      this.currentTime = time;
      this.updateProgress();
    }

    /**
     * Update progress bar and time display.
     */
    updateProgress() {
      if (!this.audio || !this.progressBar) {
        return;
      }

      this.currentTime = this.audio.currentTime;
      const percent = this.duration > 0 ? (this.currentTime / this.duration) * 100 : 0;

      const progressFill = this.progressBar.querySelector('.progress-fill');
      if (progressFill) {
        progressFill.style.width = percent + '%';
      }

      if (this.currentTimeDisplay) {
        this.currentTimeDisplay.textContent = this.formatTime(this.currentTime);
      }
      this.progressBar.setAttribute('aria-valuenow', Math.floor(this.currentTime));
    }

    /**
     * Cycle through playback speeds.
     */
    cycleSpeed() {
      const speeds = [1.0, 1.25, 1.5, 2.0];
      const currentIndex = speeds.indexOf(this.playbackSpeed);
      const nextIndex = (currentIndex + 1) % speeds.length;
      this.setPlaybackSpeed(speeds[nextIndex]);
    }

    /**
     * Set playback speed.
     */
    setPlaybackSpeed(speed) {
      this.playbackSpeed = speed;
      if (this.audio) {
        this.audio.playbackRate = speed;
      }
      this.speedButton.textContent = speed + 'x';
    }

    /**
     * Format time in MM:SS format.
     */
    formatTime(seconds) {
      if (isNaN(seconds) || seconds < 0) {
        return '0:00';
      }

      const mins = Math.floor(seconds / 60);
      const secs = Math.floor(seconds % 60);
      return mins + ':' + (secs < 10 ? '0' : '') + secs;
    }

    /**
     * Show the player.
     */
    show() {
      this.playerElement.classList.add('visible');
    }

    /**
     * Hide the player.
     */
    hide() {
      if (this.audio) {
        this.audio.pause();
      }
      this.playerElement.classList.remove('visible');
      this.userClosed = true;
      this.clearState();
      // Persist the closed state so it doesn't restore on next page.
      try {
        localStorage.setItem('audioPlayerClosed', 'true');
      } catch (e) {
        // Ignore localStorage errors.
      }
    }

    /**
     * Save player state to localStorage.
     */
    saveState() {
      // Don't save state if user explicitly closed the player.
      if (this.userClosed) {
        return;
      }

      if (!this.audio || !this.audio.src) {
        return;
      }

      const state = {
        src: this.audio.src,
        currentTime: this.audio.currentTime || 0,
        playing: !this.audio.paused,
        volume: this.audio.volume || 1,
        speed: this.playbackSpeed,
        title: this.currentTitle,
        audioId: this.currentAudioId,
        timestamp: Date.now()
      };

      try {
        localStorage.setItem('audioPlayerState', JSON.stringify(state));
      } catch (e) {
        // Ignore localStorage errors.
      }
    }

    /**
     * Restore player state from localStorage.
     */
    restoreState() {
      if (!this.playerElement || !this.audio) {
        setTimeout(() => this.restoreState(), 100);
        return false;
      }

      // Check if user explicitly closed the player (persisted across pages).
      try {
        const wasClosed = localStorage.getItem('audioPlayerClosed') === 'true';
        if (wasClosed) {
          this.userClosed = true;
          // Clear the closed flag so it can be opened again if user clicks button.
          localStorage.removeItem('audioPlayerClosed');
          return false;
        }
      } catch (e) {
        // Ignore localStorage errors.
      }

      // Don't restore if user explicitly closed the player.
      if (this.userClosed) {
        return false;
      }

      try {
        const stateStr = localStorage.getItem('audioPlayerState');
        if (!stateStr) {
          return false;
        }

        const state = JSON.parse(stateStr);
        const maxAge = 24 * 60 * 60 * 1000;
        if (Date.now() - state.timestamp > maxAge || !state.src) {
          this.clearState();
          return false;
        }

        // Initialize audio listeners if needed.
        this.initializeAudioListeners();

        // Set audio source.
        this.audio.src = state.src;
        this.audio.volume = state.volume || 1;
        this.audio.playbackRate = state.speed || 1.0;
        this.playbackSpeed = state.speed || 1.0;
        this.currentTitle = state.title || '';
        this.currentAudioId = state.audioId || '';
        this.currentUrl = state.src;

        // Update UI.
        if (this.titleElement) {
          this.titleElement.textContent = this.currentTitle;
        }
        if (this.speedButton) {
          this.speedButton.textContent = this.playbackSpeed + 'x';
        }

        // Restore position and playing state after metadata loads.
        const restoreAfterMetadata = () => {
          if (state.currentTime > 0) {
            this.audio.currentTime = state.currentTime;
            this.currentTime = state.currentTime;
            this.updateProgress();
          }

          if (state.playing) {
            // Try to play - may be blocked by browser autoplay policy.
            this.audio.play().then(() => {
              this.isPlaying = true;
              this.updatePlayPauseButton();
            }).catch(() => {
              // Autoplay blocked - user will need to click play.
              this.isPlaying = false;
              this.updatePlayPauseButton();
            });
          }
        };

        if (this.audio.readyState >= 1) {
          restoreAfterMetadata();
        } else {
          this.audio.addEventListener('loadedmetadata', restoreAfterMetadata, { once: true });
        }

        this.show();
        return true;
      } catch (e) {
        this.clearState();
        return false;
      }
    }

    /**
     * Clear saved player state.
     */
    clearState() {
      try {
        localStorage.removeItem('audioPlayerState');
        localStorage.removeItem('audioPlayerClosed');
      } catch (e) {
        // Ignore errors.
      }
    }

    /**
     * Ensure player element is outside swap target.
     */
    ensurePlayerOutsideTarget() {
      if (!this.playerElement) {
        return;
      }
      
      const target = document.querySelector(this.contentSelector);
      if (target && target.contains(this.playerElement)) {
        document.body.appendChild(this.playerElement);
      } else if (!document.body.contains(this.playerElement)) {
        document.body.appendChild(this.playerElement);
      }
    }

    /**
     * Adjust player position for admin sidebars.
     */
    adjustForSidebar() {
      if (!this.playerElement) {
        return;
      }

      const elements = document.querySelectorAll('nav, aside, [role="navigation"]');
      for (let el of elements) {
        const rect = el.getBoundingClientRect();
        const style = window.getComputedStyle(el);
        if ((style.position === 'fixed' || style.position === 'absolute') && 
            rect.left === 0 && rect.width > 0 && rect.width < 600) {
          this.playerElement.style.left = rect.width + 'px';
          return;
        }
      }
      this.playerElement.style.left = '0';
    }

    /**
     * Initialize HTMX navigation.
     */
    initHtmxNavigation(enabled, contentSelector, excludeSelectors) {
      if (this.htmxInitialized) {
        return;
      }

      // Check if htmx is available.
      if (typeof htmx === 'undefined' || !htmx) {
        console.warn('HTMX is not available. Navigation persistence will not work.');
        return;
      }

      this.navigationEnabled = enabled;
      if (contentSelector) {
        this.contentSelector = contentSelector;
      }
      if (excludeSelectors) {
        this.excludeSelectors = excludeSelectors;
      }

      if (!this.navigationEnabled) {
        return;
      }

      // Ensure player and audio are outside swap target before any request.
      htmx.on('htmx:configRequest', () => {
        this.ensurePlayerOutsideTarget();
        if (this.audio && !document.body.contains(this.audio)) {
          document.body.appendChild(this.audio);
        }
      });

      // Save state before navigation.
      htmx.on('htmx:beforeRequest', () => {
        this.saveState();
      });

      // Ensure player stays outside swap target.
      htmx.on('htmx:beforeSwap', () => {
        this.ensurePlayerOutsideTarget();
      });

      // After swap, reattach behaviors and ensure player persists.
      htmx.on('htmx:afterSwap', (event) => {
        this.ensurePlayerOutsideTarget();
        
        // Ensure audio element persists.
        if (!document.body.contains(this.audio)) {
          document.body.appendChild(this.audio);
        }

        // Resume playback if it was playing.
        if (this.audio && this.isPlaying && this.audio.paused) {
          this.audio.play().catch(() => {});
        }

        // Update title from swapped content.
        const titleElement = event.detail.target.querySelector('title');
        if (titleElement) {
          document.title = titleElement.textContent;
        }

        Drupal.attachBehaviors(event.detail.target, drupalSettings);
        this.adjustForSidebar();
        
        if (htmx.process) {
          htmx.process(event.detail.target);
        }

        // Only show player if it wasn't explicitly closed by user.
        if (this.playerElement && this.audio && this.audio.src && !this.userClosed) {
          this.show();
        }
      });

      // Intercept link clicks - only add once.
      if (!this.htmxClickHandlerAdded) {
        document.addEventListener('click', (e) => {
          const link = e.target.closest('a');
          if (!link || !link.href || this.shouldExcludeLink(link) || !this.isInternalLink(link.href)) {
            return;
          }

          const target = document.querySelector(this.contentSelector);
          if (!target) {
            return;
          }

          e.preventDefault();
          e.stopPropagation();

          if (this.audio) {
            this.saveState();
          }

          htmx.ajax('GET', link.href, {
            target: target,
            swap: 'outerHTML',
            select: this.contentSelector
          });
          
          history.pushState({}, '', link.href);
        }, true);
        
        this.htmxClickHandlerAdded = true;
      }

      // Handle browser back/forward buttons.
      window.addEventListener('popstate', () => {
        const target = document.querySelector(this.contentSelector);
        if (target) {
          htmx.ajax('GET', window.location.href, {
            target: target,
            swap: 'outerHTML',
            select: this.contentSelector
          });
        }
      });

      this.htmxInitialized = true;
    }

    /**
     * Check if link should be excluded from HTMX navigation.
     */
    shouldExcludeLink(link) {
      // Check for data-no-htmx attribute.
      if (link.hasAttribute('data-no-htmx')) {
        return true;
      }

      // Check exclude selectors.
      const excludeList = this.excludeSelectors.split(',').map(s => s.trim());
      for (let selector of excludeList) {
        if (link.matches(selector)) {
          return true;
        }
      }

      return false;
    }

    /**
     * Check if link is internal (same domain).
     */
    isInternalLink(href) {
      try {
        const linkUrl = new URL(href, window.location.href);
        return linkUrl.origin === window.location.origin;
      } catch (e) {
        // If URL parsing fails, assume it's internal if it's relative.
        return href.startsWith('/') || href.startsWith('./') || href.startsWith('../');
      }
    }
  }

  // Singleton instance.
  let playerInstance = null;

  /**
   * Drupal behavior to attach click handlers.
   */
  Drupal.behaviors.stickyAudioPlayer = {
    attach: function (context, settings) {
      if (!playerInstance) {
        playerInstance = new StickyAudioPlayer();
      }

      const playerSettings = (settings && settings.stickyAudioPlayer) 
        || (typeof drupalSettings !== 'undefined' && drupalSettings.stickyAudioPlayer)
        || null;

      // Initialize HTMX navigation if enabled.
      if (playerSettings && playerSettings.navigationEnabled && !playerInstance.htmxInitialized) {
        const initHtmx = () => {
          if (typeof htmx !== 'undefined' && htmx) {
            playerInstance.initHtmxNavigation(
              playerSettings.navigationEnabled,
              playerSettings.contentSelector,
              playerSettings.excludeSelectors
            );
          } else {
            setTimeout(initHtmx, 100);
          }
        };
        initHtmx();
      }

      // Restore state once on initial load.
      if (!playerInstance.stateRestored) {
        setTimeout(() => {
          playerInstance.restoreState();
          playerInstance.stateRestored = true;
        }, 200);
      }

      // Attach click handlers to buttons.
      context.querySelectorAll('.listen-to-article-btn:not([data-audio-player-attached])').forEach((button) => {
        button.setAttribute('data-audio-player-attached', 'true');
        button.addEventListener('click', function (e) {
          e.preventDefault();
          const audioId = this.getAttribute('data-audio-id');
          const audioUrl = this.getAttribute('data-audio-url');
          const articleTitle = this.getAttribute('data-article-title');

          if (audioUrl) {
            const autostart = playerSettings && playerSettings[audioId] && playerSettings[audioId].autostart;
            playerInstance.loadAudio(audioUrl, articleTitle, audioId, autostart);
          }
        });
      });
    }
  };

})(Drupal, drupalSettings, typeof htmx !== 'undefined' ? htmx : undefined);

