<?php

namespace Drupal\straker_translate\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\straker_translate\LanguageLocaleMapperInterface;
use Drupal\straker_translate\StrakerTranslateInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for redirecting to the Straker Translate Verify workbench.
 *
 * @package Drupal\straker_translate\Controller
 */
class StrakerTranslateWorkbenchRedirectController extends StrakerTranslateControllerBase {

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a StrakerTranslateControllerBase object.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The Request instance.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\straker_translate\StrakerTranslateInterface $straker_translate
   *   The straker_translate service.
   * @param \Drupal\straker_translate\LanguageLocaleMapperInterface $language_locale_mapper
   *   The language-locale mapper.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(Request $request, ConfigFactoryInterface $config_factory, StrakerTranslateInterface $straker_translate, LanguageLocaleMapperInterface $language_locale_mapper, FormBuilderInterface $form_builder, LoggerInterface $logger, TimeInterface $time = NULL) {
    parent::__construct($request, $config_factory, $straker_translate, $language_locale_mapper, $form_builder, $logger);
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('config.factory'),
      $container->get('straker_translate'),
      $container->get('straker_translate.language_locale_mapper'),
      $container->get('form_builder'),
      $container->get('logger.channel.straker_translate'),
      $container->get('datetime.time')
    );
  }

  /**
   * Redirects to the workbench of the given document and locale in the Verify.
   */
  public function redirectToWorkbench($doc_id) {
    // Get account settings to build workbench link.
    $account = $this->config('straker_translate.account');
    // Generate the uri to the Straker Translate Workbench.
    $uri = $this->generateWorkbenchUri($doc_id, $account);
    return new TrustedRedirectResponse(Url::fromUri($uri)->toString());
  }

  /**
   * Generates a workbench uri for this account given a document id and locale.
   *
   * @param string $document_id
   *   The document id.
   * @param string $locale
   *   Straker translation language.
   * @param \Drupal\Core\Config\Config $account
   *   Config with the account settings.
   *
   * @return string
   *   The uri of the workbench for this account for editing this translation.
   */
  protected function generateWorkbenchUri($document_id, $account) {
    $base_url = $account->get('domain');
    // https://{environment}/workbench/document/{uuid}/locale/{es-MX}
    $workbench_uri = $base_url . '/projects/files/' . $document_id;
    return $workbench_uri;
  }

}
