<?php

namespace Drupal\straker_translate\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\straker_translate\Plugin\Field\FieldType\StrakerTranslateTranslationSourceField;

/**
 * Defines the Straker Translate content metadata entity.
 *
 * Saves the metadata of content entities.
 *
 * @ContentEntityType(
 *   id = "straker_content_metadata",
 *   label = @Translation("Straker Translate Content Metadata"),
 *   base_table = "straker_translate_metadata",
 *   entity_keys = {
 *     "id" = "id"
 *   },
 *   handlers = {
 *     "views_data" = "\Drupal\straker_translate\Views\StrakerTranslateContentMetadataViewsData",
 *   }
 * )
 */
class StrakerTranslateContentMetadata extends ContentEntityBase {

  /**
   * @{inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['content_entity_type_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Content entity type ID'))
      ->setDescription(new TranslatableMarkup('The ID of the content entity type this Straker Translate status is for.'))
      ->setRequired(TRUE);

    $fields['content_entity_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Content entity ID'))
      ->setDescription(new TranslatableMarkup('The ID of the content entity this Straker Translate status is for.'))
      ->setRequired(TRUE);

    $fields['document_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Straker Translate document id'))
      ->setDescription(new TranslatableMarkup('The Straker Translate document id.'));

    $fields['hash'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Straker Translate hash'))
      ->setDescription(new TranslatableMarkup('A hash of the Straker Translate saved entity data, required for checking for changes.'));

    $fields['profile'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Straker Translate profile'))
      ->setDescription(new TranslatableMarkup('The Straker Translate profile defining this translation.'))
      ->setSetting('target_type', 'straker_translate_profile');

    $fields['translation_source'] = BaseFieldDefinition::create('language')
      ->setComputed(TRUE)
      ->setClass(StrakerTranslateTranslationSourceField::class)
      ->setLabel(new TranslatableMarkup('Straker translation source'))
      ->setDescription(new TranslatableMarkup('The source language from which this translation was created.'))
      ->setDefaultValue(LanguageInterface::LANGCODE_NOT_SPECIFIED)
      ->setTranslatable(TRUE);

    $fields['translation_status'] = BaseFieldDefinition::create('straker_translate_language_key_value')
      ->setLabel(new TranslatableMarkup('Straker translation status'))
      ->setDescription(new TranslatableMarkup('The status of the source in case of being the source translation, or the status of the translation.'))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED);

    $fields['job_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Straker Translate job id'))
      ->setDescription(new TranslatableMarkup('The Straker Translate job id.'));

    $fields['updated_timestamp'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Updated timestamp'))
      ->setDescription(new TranslatableMarkup('The last time document was updated.'));

    $fields['uploaded_timestamp'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Initial upload'))
      ->setDescription(new TranslatableMarkup('The time of the initial upload.'));

    return $fields;
  }

  /**
   * Loads a Straker Translate content metadata entity based on the entity type and id.
   *
   * @param string $target_entity_type_id
   *   Target entity type id.
   * @param int $target_id
   *   Entity id.
   *
   * @return $this
   *   The Straker Translate content metadata if one exists. Otherwise, returns
   *   default values.
   */
  public static function loadByTargetId($target_entity_type_id, $target_id) {
    $metadata = NULL;
    if ($target_entity_type_id == NULL || $target_id == NULL) {
      return NULL;
    }
    $entity_query = \Drupal::entityQuery('straker_content_metadata');
    $entity_query->accessCheck(FALSE)->condition('content_entity_type_id', $target_entity_type_id)
      ->condition('content_entity_id', $target_id);
    $result = $entity_query->execute();
    if (!empty($result)) {
      $metadata = self::load(reset($result));
    }

    if ($metadata === NULL) {
      $metadata = new static(
        ['content_entity_type_id' => $target_entity_type_id, 'content_entity_id' => $target_id], 'straker_content_metadata');
    }
    return $metadata;
  }

  /**
   * Loads a Straker Translate content metadata entity based on the entity type and id.
   *
   * @param string $document_id
   *   Straker Translate Document ID.
   *
   * @return $this|null
   *   The Straker Translate content metadata if it exists. Otherwise, returns NULL.
   */
  public static function loadByDocumentId($document_id) {
    $metadata = NULL;
    if ($document_id !== NULL) {
      $entity_query = \Drupal::entityQuery('straker_content_metadata');
      $entity_query->accessCheck(FALSE);
      $entity_query->condition('document_id', $document_id);
      $result = $entity_query->execute();
      if (!empty($result)) {
        $metadata = self::load(reset($result));
      }
    }
    return $metadata;
  }

  /**
   * Loads all Straker Translate document IDs stored in the system.
   *
   * @return string[]
   *   Indexed array of all the document ids.
   */
  public static function getAllLocalDocumentIds() {
    return \Drupal::database()->select('straker_translate_metadata', 'lcm')
      ->fields('lcm', ['document_id'])
      ->execute()
      ->fetchCol(0);
  }

  /**
   * Gets the Straker Translate document id.
   *
   * @return string
   *   The Straker Translate document ID.
   */
  public function getDocumentId() {
    return $this->document_id->value;
  }

  /**
   * Sets the Straker Translate document id.
   *
   * @param string $document_id
   *   Straker Translate Document ID.
   */
  public function setDocumentId($document_id) {
    $this->document_id->value = $document_id;
  }

  /**
   * Gets the Straker Translate profile.
   *
   * @return string
   *   The profile name.
   */
  public function getProfile() {
    return $this->profile->target_id;
  }

  /**
   * Sets the Straker Translate profile.
   *
   * @param string $profile_id
   *   The profile name.
   */
  public function setProfile($profile_id) {
    $this->profile->target_id = $profile_id;
  }

  /**
   * Sets the content entity this Straker Translate metadata relates to.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The content entity.
   *
   * @return $this
   */
  public function setEntity(EntityInterface $entity) {
    $this->content_entity_type_id->value = $entity->getEntityTypeId();
    $this->content_entity_id->value = $entity->id();
    return $this;
  }

  /**
   * Gets the content entity type ID.
   *
   * @return string
   *   The content entity type ID.
   */
  public function getContentEntityTypeId() {
    return $this->content_entity_type_id->value;
  }

  /**
   * Gets the content entity ID.
   *
   * @return string
   *   The content entity ID.
   */
  public function getContentEntityId() {
    return $this->content_entity_id->value;
  }

  /**
   * Gets the Job ID.
   *
   * @return string
   *   The job ID.
   */
  public function getJobId() {
    return $this->job_id->value;
  }

  /**
   * Sets the content entity type ID.
   *
   * @param string $value
   *   The content entity type ID.
   *
   * @return $this
   */
  public function setContentEntityTypeId($value) {
    $this->content_entity_type_id = $value;
    return $this;
  }

  /**
   * Sets the content entity ID.
   *
   * @param string $value
   *   The content entity ID.
   *
   * @return $this
   */
  public function setContentEntityId($value) {
    $this->content_entity_id = $value;
    return $this;
  }

  /**
   * Sets the job ID.
   *
   * @param string $value
   *   The job ID.
   *
   * @return $this
   */
  public function setJobId($value) {
    $this->job_id = $value;
    return $this;
  }

  /**
   * Sets the timestamp for the time of the initial upload.
   *
   * @param int $timestamp
   *   The unix timestamp of the event.
   *
   * @return $this
   */
  public function setLastUpdated($timestamp) {
    $this->updated_timestamp->value = $timestamp;

    return $this;
  }

  /**
   * Gets the timestamp for the time of the initial upload.
   *
   * @return int
   */
  public function getLastUpdated() {
    return $this->updated_timestamp->value;
  }

  /**
   * Sets the timestamp for the time of the initial upload.
   *
   * @param int $timestamp
   *   The unix timestamp of the event.
   *
   * @return $this
   */
  public function setLastUploaded($timestamp) {
    $this->uploaded_timestamp->value = $timestamp;
    return $this;
  }

  /**
   * Gets the timestamp for the last time document was updated.
   *
   * @return int
   */
  public function getLastUploaded() {
    return $this->uploaded_timestamp->value;
  }

}
