<?php

namespace Drupal\straker_translate\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\straker_translate\StrakerTranslateConfigTranslationServiceInterface;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Straker Translate.
 */
class StrakerTranslateSettingsTabConfigurationForm extends StrakerTranslateConfigFormBase {

  protected $profile_options;
  protected $profiles;
  protected $bundles;
  protected $translatable_bundles;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The Straker Translate configuration service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationService
   */
  protected $straker_translateConfig;

  /**
   * The Straker Translate config translation service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigTranslationServiceInterface
   */
  protected $translationService;

  /**
   * A array of configuration mapper instances.
   *
   * @var \Drupal\config_translation\ConfigMapperInterface[]
   */
  protected $mappers;

  /**
   * The Entity Type Manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new StrakerTranslateManagementForm object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_config
   *   The Straker Translate config service.
   * @param \Drupal\straker_translate\StrakerTranslateConfigTranslationServiceInterface $translation_service
   *   The Straker Translate config translation service.
   * @param \Drupal\config_translation\ConfigMapperInterface[] $mappers
   *   The configuration mapper manager.
   */
  public function __construct(LanguageManagerInterface $language_manager, EntityTypeManagerInterface $entity_type_manager, StrakerTranslateConfigurationServiceInterface $straker_translate_config, StrakerTranslateConfigTranslationServiceInterface $translation_service, array $mappers) {
    $this->languageManager = $language_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->straker_translateConfig = $straker_translate_config;
    $this->translationService = $translation_service;
    $this->mappers = $mappers;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('entity_type.manager'),
      $container->get('straker_translate.configuration'),
      $container->get('straker_translate.config_translation'),
      $container->get('plugin.manager.config_translation.mapper')->getMappers()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormID() {
    return 'straker_translate.settings_tab_configuration_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $profile_options = $this->straker_translateConfig->getProfileOptions();

    $header = [
      'enabled' => $this->t('Enable'),
      'type' => $this->t('Configuration Type'),
      'profile' => $this->t('Translation Profile'),
    ];

    $table = [
      '#type' => 'table',
      '#header' => $header,
      '#empty' => $this->t('No Entries'),
    ];

    foreach ($this->mappers as $mapper) {
      // We don't want to show config objects, where we only have one instance.
      // Just show config entities.
      $enabled = $this->translationService->isEnabled($mapper->getPluginId());
      $row = [];
      $row['enabled'] = [
        '#type' => 'checkbox',
        '#default_value' => $enabled,
      ];
      $row['type'] = [
        '#markup' => $mapper->getTypeLabel(),
      ];
      $row['profile'] = [
        '#type' => 'select',
        '#options' => $this->straker_translateConfig->getProfileOptions(),
        '#default_value' => $this->straker_translateConfig->getConfigEntityDefaultProfileId($mapper->getPluginId()),
      ];
      $table[$mapper->getPluginId()] = $row;
    }
    ksort($table);

    $form['config'] = [
      '#type' => 'details',
      '#title' => 'Translate Configuration Types',
    ];

    $form['config']['table'] = $table;

    $form['config']['actions']['#type'] = 'actions';
    $form['config']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue(['table']);
    foreach ($values as $plugin_id => $data) {
      // We only save the enabled status if it changed.
      if ($data['enabled'] && !$this->translationService->isEnabled($plugin_id)) {
        $this->translationService->setEnabled($plugin_id, TRUE);
      }
      if (!$data['enabled'] && $this->translationService->isEnabled($plugin_id)) {
        $this->translationService->setEnabled($plugin_id, FALSE);
      }
      // If we enable it, we save the profile.
      if ($data['enabled'] && $data['profile'] !== $this->straker_translateConfig->getConfigEntityDefaultProfileId($plugin_id, FALSE)) {
        $this->straker_translateConfig->setConfigEntityDefaultProfileId($plugin_id, $data['profile']);
      }
    }
  }

}
