<?php

namespace Drupal\straker_translate\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic menu links for Straker Translate content translation management.
 */
class ContentTranslationMenuLinks extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The base plugin ID.
   *
   * @var string
   */
  protected $basePluginId;

  /**
   * The Straker Translate configuration service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface
   */
  protected $straker_translateConfiguration;

  /**
   * Constructs a new ContentTranslationLocalTasks.
   *
   * @param string $base_plugin_id
   *   The base plugin ID.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The Straker Translate configuration service.
   */
  public function __construct($base_plugin_id, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration) {
    $this->basePluginId = $base_plugin_id;
    $this->straker_translateConfiguration = $straker_translate_configuration;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $base_plugin_id,
      $container->get('straker_translate.configuration')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    // Create menu links for all possible entity types.
    foreach ($this->straker_translateConfiguration->getEnabledEntityTypes() as $entity_type_id => $entity_type) {
      $translation_route_name = "straker_translate.manage.$entity_type_id";
      // If it's a paragraph, we depend on the preference.
      if ($entity_type_id === 'paragraph') {
        $config = \Drupal::config('straker_translate.settings');
        $enable_bulk_management = $config->get('preference.contrib.paragraphs.enable_bulk_management') ?? FALSE;
        if (!$enable_bulk_management) {
          // If already existed remove it.
          if (isset($this->derivatives[$translation_route_name])) {
            unset($this->derivatives[$translation_route_name]);
          }
          continue;
        }
      }
      if ($entity_type_id === 'cohesion_layout') {
        if (isset($this->derivatives[$translation_route_name])) {
          unset($this->derivatives[$translation_route_name]);
        }
        continue;
      }

      // Create the entries for the tabs in the bulk manage pages.
      $base_route_name = "straker_translate.manage";
      $this->derivatives[$translation_route_name] = [
        'entity_type_id' => $entity_type_id,
        'title' => $entity_type->getLabel(),
        'route_name' => $translation_route_name,
        'base_route' => $base_route_name,
        'parent' => 'straker_translate.manage',
      ] + $base_plugin_definition;
    }

    return parent::getDerivativeDefinitions($base_plugin_definition);
  }

}
