<?php

namespace Drupal\straker_translate\Plugin\StrakerTranslateFieldProcessor;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\straker_translate\FieldProcessor\StrakerTranslateFieldProcessorInterface;
use Drupal\straker_translate\StrakerTranslateConfigTranslationServiceInterface;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\straker_translate\StrakerTranslateContentTranslationEntityRevisionResolver;
use Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @StrakerTranslateFieldProcessor(
 *   id = "entity_reference",
 *   weight = 5,
 * )
 */
class StrakerTranslateEntityReferenceProcessor extends PluginBase implements StrakerTranslateFieldProcessorInterface, ContainerFactoryPluginInterface {

  /**
   * The Straker Translate configuration service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface
   */
  protected $straker_translateConfiguration;

  /**
   * The Straker Translate configuration translation service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateConfigTranslationService
   */
  protected $straker_translateConfigTranslation;

  /**
   * The Straker Translate content translation service.
   *
   * @var \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface
   */
  protected $straker_translateContentTranslation;

  /**
   * The entity manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new UploadToStrakerTranslateAction action.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The Straker Translate configuration service.
   * @param \Drupal\straker_translate\StrakerTranslateConfigTranslationServiceInterface $straker_translate_config_translation
   *   The Straker Translate config translation service.
   * @param \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface $straker_translate_content_translation
   *   The Straker Translate content translation service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, StrakerTranslateConfigTranslationServiceInterface $straker_translate_config_translation, StrakerTranslateContentTranslationServiceInterface $straker_translate_content_translation) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->straker_translateConfiguration = $straker_translate_configuration;
    $this->straker_translateConfigTranslation = $straker_translate_config_translation;
    $this->straker_translateContentTranslation = $straker_translate_content_translation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('straker_translate.configuration'),
      $container->get('straker_translate.config_translation'),
      $container->get('straker_translate.content_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function appliesToField(FieldDefinitionInterface $field_definition, ContentEntityInterface &$entity) {
    return in_array($field_definition->getType(), ['entity_reference', 'er_viewmode', 'bricks']);
  }

  /**
   * {@inheritdoc}
   */
  public function extract(ContentEntityInterface &$entity, string $field_name, FieldDefinitionInterface $field_definition, array &$data, array &$visited = [], string $revision_mode = StrakerTranslateContentTranslationEntityRevisionResolver::RESOLVE_LATEST_TRANSLATION_AFFECTED) {
    $target_entity_type_id = $field_definition->getFieldStorageDefinition()->getSetting('target_type');
    foreach ($entity->get($field_name) as $delta => $field_item) {
      $embedded_entity_id = $field_item->get('target_id')->getValue();
      $embedded_entity = $this->entityTypeManager->getStorage($target_entity_type_id)->load($embedded_entity_id);
      // We may have orphan references, so ensure that they exist before
      // continuing.
      if ($embedded_entity !== NULL) {
        if ($embedded_entity instanceof ContentEntityInterface) {
          // We need to avoid cycles if we have several entity references
          // referencing each other.
          if (!isset($visited[$embedded_entity->bundle()]) || !in_array($embedded_entity->id(), $visited[$embedded_entity->bundle()])) {
            $embedded_data = $this->straker_translateContentTranslation->getSourceData($embedded_entity, $visited, $revision_mode);
            $data[$field_name][$delta] = $embedded_data;
          }
          else {
            // We don't want to embed the data, but still will need the
            // references, so let's include the metadata.
            $metadata = [];
            $this->includeMetadata($embedded_entity, $metadata);
            $data[$field_name][$delta] = $metadata;
          }
        }
        elseif ($embedded_entity instanceof ConfigEntityInterface) {
          $embedded_data = $this->straker_translateConfigTranslation->getSourceData($embedded_entity);
          $data[$field_name][$delta] = $embedded_data;
        }
      }
      else {
        // If the referenced entity doesn't exist, remove the target_id
        // that may be already set.
        unset($data[$field_name]);
      }
    }
  }

  /**
   *
   */
  protected function includeMetadata(ContentEntityInterface &$entity, &$data) {
    $data['_straker_translate_metadata']['_entity_type_id'] = $entity->getEntityTypeId();
    $data['_straker_translate_metadata']['_entity_id'] = $entity->id();
    $data['_straker_translate_metadata']['_entity_revision'] = $entity->getRevisionId();
  }

  /**
   * {@inheritdoc}
   */
  public function store(ContentEntityInterface &$translation, string $langcode, ContentEntityInterface &$revision, string $field_name, FieldDefinitionInterface $field_definition, array &$field_data) {
    $target_entity_type_id = $field_definition->getFieldStorageDefinition()
      ->getSetting('target_type');
    $translation->{$field_name} = NULL;
    $delta = 0;
    foreach ($field_data as $index => $field_item) {
      if (isset($field_item['_straker_translate_metadata'])) {
        $target_entity_type_id = $field_item['_straker_translate_metadata']['_entity_type_id'];
        $embedded_entity_id = $field_item['_straker_translate_metadata']['_entity_id'];
        $embedded_entity_revision_id = $field_item['_straker_translate_metadata']['_entity_revision'];
      }
      else {
        // Try to get it from the revision itself. It may have been
        // modified, so this can be a source of errors, but we need this
        // because we didn't have metadata before.
        $embedded_entity_id = $revision->{$field_name}->get($index)
          ->get('target_id')
          ->getValue();
      }
      $embedded_entity = $this->entityTypeManager->getStorage($target_entity_type_id)
        ->load($embedded_entity_id);
      // We may have orphan references, so ensure that they exist before
      // continuing.
      if ($embedded_entity !== NULL) {
        // @todo It can be a content entity, or a config entity.
        if ($embedded_entity instanceof ContentEntityInterface) {
          if ($this->straker_translateConfiguration->isEnabled($embedded_entity->getEntityTypeId(), $embedded_entity->bundle())) {
            $this->straker_translateContentTranslation->saveTargetData($embedded_entity, $langcode, $field_item);
          }
          else {
            \Drupal::logger('straker_translate')->warning('Field %field not saved as its referenced entity is not translatable by Straker Translate', ['%field' => $field_name]);
          }
        }
        elseif ($embedded_entity instanceof ConfigEntityInterface) {
          $this->straker_translateConfigTranslation->saveTargetData($embedded_entity, $langcode, $field_item);
        }
        // Now the embedded entity is saved, but we need to ensure
        // the reference will be saved too.
        $translation->get($field_name)->set($delta, $embedded_entity_id);
        $delta++;
      }
    }
  }

}
