<?php

namespace Drupal\straker_translate\Plugin\StrakerTranslateFormComponent\BulkAction;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\straker_translate\Exception\StrakerTranslateApiException;
use Drupal\straker_translate\FormComponent\StrakerTranslateFormComponentBulkActionBase;
use Drupal\straker_translate\FormComponent\StrakerTranslateFormComponentBulkActionExecutor;
use Drupal\straker_translate\LanguageLocaleMapperInterface;
use Drupal\straker_translate\StrakerTranslate;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a Straker Translate bulk action plugin for the request operation for a single
 * language.
 *
 * @StrakerTranslateFormComponentBulkAction(
 *   id = "change_profile",
 *   title = @Translation("Change to %profile Profile"),
 *   group = @Translation("Change Translation Profile"),
 *   weight = 100,
 *   form_ids = {
 *     "straker_translate_management",
 *     "straker_translate_entity_management",
 *   },
 *   batch = {
 *     "title" = @Translation("Updating Translation Profile.")
 *   },
 *   deriver = "Drupal\straker_translate\Plugin\StrakerTranslateFormComponent\BulkAction\Derivative\ProfileStrakerTranslateBulkActionDeriver",
 * )
 */
class AssignProfile extends StrakerTranslateFormComponentBulkActionBase {

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The language-locale mapper.
   *
   * @var \Drupal\straker_translate\LanguageLocaleMapperInterface
   */
  protected $languageLocaleMapper;

  /**
   * AssignProfile constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity_type.manager service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language_manager service.
   * @param \Drupal\straker_translate\LanguageLocaleMapperInterface $language_locale_mapper
   *   The language-locale mapper.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The straker_translate.configuration service.
   * @param \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface $translation_service
   *   The straker_translate.content_translation service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity_type.bundle.info service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, LanguageLocaleMapperInterface $language_locale_mapper, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, StrakerTranslateContentTranslationServiceInterface $translation_service, EntityTypeBundleInfoInterface $entity_type_bundle_info) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $language_manager, $straker_translate_configuration, $translation_service);
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->languageLocaleMapper = $language_locale_mapper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('straker_translate.language_locale_mapper'),
      $container->get('straker_translate.configuration'),
      $container->get('straker_translate.content_translation'),
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   *
   */
  public function executeSingle(ContentEntityInterface $entity, $options, StrakerTranslateFormComponentBulkActionExecutor $executor, array &$context) {
    $profile_id = $this->pluginDefinition['profile_id'];
    $context['message'] = $this->t('Changing Translation Profile for @type %label.', ['@type' => $entity->getEntityType()->getLabel(), '%label' => $entity->label()]);
    $bundleInfos = $this->entityTypeBundleInfo->getBundleInfo($entity->getEntityTypeId());
    if (!$entity->getEntityType()->isTranslatable() || !$bundleInfos[$entity->bundle()]['translatable']) {
      $this->messenger()->addWarning($this->t('Cannot change profile for @type %label. That @bundle_label is not enabled for translation.',
        ['@type' => $bundleInfos[$entity->bundle()]['label'], '%label' => $entity->label(), '@bundle_label' => $entity->getEntityType()->getBundleLabel()]));
      return FALSE;
    }
    if (!$this->straker_translateConfiguration->isEnabled($entity->getEntityTypeId(), $entity->bundle())) {
      $this->messenger()->addWarning($this->t('Cannot change profile for @type %label. That @bundle_label is not enabled for Straker translation.',
        ['@type' => $bundleInfos[$entity->bundle()]['label'], '%label' => $entity->label(), '@bundle_label' => $entity->getEntityType()->getBundleLabel()]
      ));
      return FALSE;
    }
    try {
      $this->straker_translateConfiguration->setProfile($entity, $profile_id, TRUE);
      if ($profile_id === StrakerTranslate::PROFILE_DISABLED) {
        $this->translationService->setSourceStatus($entity, StrakerTranslate::STATUS_DISABLED);
        $this->translationService->setTargetStatuses($entity, StrakerTranslate::STATUS_DISABLED);
      }
      elseif ($this->translationService->getSourceStatus($entity) === StrakerTranslate::STATUS_DISABLED) {
        if ($this->translationService->getDocumentId($entity) !== NULL) {
          $this->translationService->setSourceStatus($entity, StrakerTranslate::STATUS_CURRENT);
        }
        else {
          $this->translationService->setSourceStatus($entity, StrakerTranslate::STATUS_CURRENT);
        }
        $this->translationService->checkTargetStatuses($entity);
      }
    }
    catch (StrakerTranslateApiException $exception) {
      $this->messenger()->addError($this->t('The Translation Profile change for @entity_type %title failed. Please try again.', ['@entity_type' => $entity->getEntityTypeId(), '%title' => $entity->label()]));
      return FALSE;
    }
    return TRUE;
  }

}
