<?php

namespace Drupal\straker_translate\Plugin\StrakerTranslateFormComponent\Field;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\straker_translate\FormComponent\StrakerTranslateFormComponentBundleTrait;
use Drupal\straker_translate\FormComponent\StrakerTranslateFormComponentFieldBase;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a Straker Translate form-field plugin for an entity's bundle.
 *
 * @StrakerTranslateFormComponentField(
 *   id = "bundle",
 *   title = @Translation("Bundle"),
 *   weight = 100,
 *   form_ids = {
 *     "straker_translate_management",
 *     "straker_translate_entity_management",
 *   }
 * )
 */
class Bundle extends StrakerTranslateFormComponentFieldBase {

  use StrakerTranslateFormComponentBundleTrait;

  /**
   * StrakerTranslateFormComponentBase constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity_type.manager service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language_manager service.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The straker_translate.configuration service.
   * @param \Drupal\straker_translate\StrakerTranslateContentTranslationServiceInterface $translation_service
   *   The straker_translate.content_translation service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, StrakerTranslateContentTranslationServiceInterface $translation_service, EntityTypeBundleInfoInterface $entity_type_bundle_info) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $language_manager, $straker_translate_configuration, $translation_service);
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('straker_translate.configuration'),
      $container->get('straker_translate.content_translation'),
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(array $arguments = []) {
    $entity_type_id = $arguments['entity_type_id'] ?? NULL;
    // If we are showing multiple entity types, we default to show the bundle
    // field.
    return $entity_type_id ? $this->hasBundles($entity_type_id) : TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeader($entity_type_id = NULL) {
    if ($entity_type_id === NULL) {
      return $this->t('Bundle');
    }
    $entity_type = $this->getEntityType($entity_type_id);
    return ['data' => $entity_type->getBundleLabel()];
  }

  /**
   * {@inheritdoc}
   */
  public function getData(EntityInterface $entity) {
    if ($bundle_info = $this->getBundleInfo($entity->getEntityTypeId()) ?? []) {
      $data = $bundle_info[$entity->bundle()]['label'] ?? '';
    }

    return $data ?? '';
  }

}
