<?php

namespace Drupal\straker_translate\Remote;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\straker_translate\Exception\StrakerTranslateApiException;
use Drupal\straker_translate\Exception\StrakerTranslateDocumentNotFoundException;
use Drupal\straker_translate\Exception\StrakerTranslateProcessedWordsLimitException;
use GuzzleHttp\Exception\ClientException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * A simple connector to the Straker translation API.
 */
class StrakerTranslateApi implements StrakerTranslateApiInterface {

  /**
   * The HTTP client to interact with the Straker Translate service.
   *
   * @var \Drupal\straker_translate\Remote\StrakerTranslateHttpInterface
   */
  protected $straker_translateClient;

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a StrakerTranslateApi object.
   *
   * @param \Drupal\straker_translate\Remote\StrakerTranslateHttpInterface $client
   *   A http client.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   */
  public function __construct(StrakerTranslateHttpInterface $client, LoggerInterface $logger) {
    $this->straker_translateClient = $client;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('straker_translate.http_client'),
      $container->get('logger.channel.straker_translate')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getLocales($force_refresh = FALSE) {
    $cid = 'straker_translate.locales';
    if (!$force_refresh) {
      $cache = \Drupal::cache()->get($cid);
      if ($cache) {
        return $cache->data;
      }
    }
    $this->logger->debug('Starting Locales request: /languages');
    try {
      /** @var \Psr\Http\Message\ResponseInterface $response */
      $response = $this->straker_translateClient->get('/languages');
      if ($response->getStatusCode() == Response::HTTP_OK) {
        $data = json_decode($response->getBody(), TRUE);
        $this->logger->debug('getLocales response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
        \Drupal::cache()->set($cid, $data);
        return $data;
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error requesting locales: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Error requesting locales: ' . $e->getMessage());
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountInfo() {
    try {
      // As we dont have account info endpoint, we use the workflows endpoint.
      // Once we have the account info endpoint, we can use it.
      $workflows = $this->getWorkflows();
    }
    catch (\Exception $e) {
      $this->logger->error('Error requesting account info: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Failed to get account info: ' . $e->getMessage());
    }
    $this->logger->debug('getAccountInfo response received');
    return $workflows;
  }

  /**
   * {@inheritdoc}
   */
  public function addDocument($args) {
    try {
      $this->logger->debug('StrakerTranslate::addDocument (POST /project) called with %args', ['%args' => var_export($args, TRUE)]);
      $response = $this->straker_translateClient->post('/project', $args, TRUE);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_TOO_MANY_REQUESTS) {
        $message = $e->getMessage();
        // This only applies to trial users.
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        if (!empty($responseBody) && isset($responseBody['messages'])) {
          $message = $responseBody['messages'][0];
        }
        throw new StrakerTranslateProcessedWordsLimitException($message);
      }
      throw new StrakerTranslateApiException('Error adding document: ' . $e->getMessage());
    }
    catch (\Exception $e) {
      $this->logger->error('Error adding document: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Error adding document: ' . $e->getMessage());
    }
    if ($response->getStatusCode() == Response::HTTP_OK) {
      $data = json_decode($response->getBody(), TRUE);
      $this->logger->debug('Project Created Successfully, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
      if (!empty($data['project_id'])) {
        return $response;
      }
    }
    // @todo log warning
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function confirmDocument($id) {
    try {
      $this->logger->debug('StrakerTranslate::confirmDocument called with id %id', ['%id' => $id]);
      $args = [
        'project_id' => $id,
      ];
      $response = $this->straker_translateClient->post('/project/confirm', $args);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to Confirm document: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error Confirming document: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to Confirm the document: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    $this->logger->debug('ConfirmDocument response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function cancelDocument($id) {
    try {
      $this->logger->debug('StrakerTranslate::cancelDocument called with id %id', ['%id' => $id]);
      $args = [
        'id' => $id,
        'cancelled_reason' => 'CANCELLED_BY_AUTHOR',
      ];
      $response = $this->straker_translateClient->post('/api/document/' . $id . '/cancel', $args);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to cancel document: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error cancelling document: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to cancel document: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    $this->logger->debug('cancelDocument response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function cancelDocumentTarget($document_id, $locale) {
    try {
      $this->logger->debug('StrakerTranslate::cancelDocumentTarget called with id %id and locale %locale', ['%id' => $document_id, '%locale' => $locale]);
      $args = [
        'id' => $document_id,
        'locale' => $locale,
        'cancelled_reason' => 'CANCELLED_BY_AUTHOR',
        'mark_invoiceable' => 'true',
      ];
      $response = $this->straker_translateClient->post('/api/document/' . $document_id . '/translation/' . $locale . '/cancel', $args);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to cancel document target: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error cancelling document target: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to cancel document target: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    $this->logger->debug('cancelDocumentTarget response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentInfo($id) {
    try {
      $this->logger->debug('StrakerTranslate::getDocumentInfo called with id %id', ['%id' => $id]);
      $response = $this->straker_translateClient->get('/api/document/' . $id);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      $this->logger->error('Error getting document info: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to get document: %message', ['%message' => $e->getMessage()]), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting document info: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException(new FormattableMarkup('Failed to get document: %message', ['%message' => $e->getMessage()]));
    }
    $this->logger->debug('getDocumentInfo response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentStatus($project_id) {
    try {
      $this->logger->debug('StrakerTranslate::getDocumentStatus called with id %id', ['%id' => $project_id]);
      $response = $this->straker_translateClient->get('/project/' . $project_id);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException('Failed to get document status: ' . $e->getMessage(), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting document status: %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Failed to get document status: ' . $e->getMessage());
    }
    $this->logger->debug('getDocumentStatus response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentTranslationStatuses($id) {
    try {
      $this->logger->debug('StrakerTranslate::getDocumentTranslationStatuses called with %id', ['%id' => $id]);
      $response = $this->straker_translateClient->get('/api/document/' . $id . '/translation');
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException('Failed to get document translation status: ' . $e->getMessage(), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting document translation status (%id): %message.',
        ['%id' => $id, '%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Failed to get document translation status: ' . $e->getMessage());
    }
    $this->logger->debug('getDocumentTranslationStatuses response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentTranslationStatus($id, $locale) {
    try {
      $this->logger->debug('StrakerTranslate::getDocumentTranslationStatus called with %id and %locale', ['%id' => $id, '%locale' => $locale]);
      $response = $this->straker_translateClient->get('/api/document/' . $id . '/translation');
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException('Failed to get document translation status: ' . $e->getMessage(), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting document translation status (%id, %locale): %message.',
        ['%id' => $id, '%locale' => $locale, '%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Failed to get document translation status: ' . $e->getMessage());
    }
    $this->logger->debug('getDocumentTranslationStatus response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function getTranslation($id, $langcode) {
    try {
      $this->logger->debug('StrakerTranslate::getTranslation called with %id and %language', ['%id' => $id, '%language' => $langcode]);
      $response = $this->straker_translateClient->get('/file/' . $id);
    }
    catch (ClientException $e) {
      if ($e->getCode() === Response::HTTP_NOT_FOUND) {
        $responseBody = json_decode($e->getResponse()->getBody(), TRUE);
        $message = $responseBody['messages'][0];
        throw new StrakerTranslateDocumentNotFoundException($message, Response::HTTP_NOT_FOUND);
      }
      throw new StrakerTranslateApiException('Failed to add translation: ' . $e->getMessage(), $e->getCode(), $e);
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting translation (%id, %language): %message.',
        ['%id' => $id, '%language' => $langcode, '%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Failed to add translation: ' . $e->getMessage());
    }
    $this->logger->debug('getTranslation response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function getWorkflows() {
    // Cache ID.
    $cid = 'straker_translate.workflows';
    $cache = \Drupal::cache()->get($cid);
    // If cache exists and not expired, return it.
    if ($cache) {
      return $cache->data;
    }
    // If cache does not exist, fetch workflows from the API.
    try {
      $this->logger->debug('StrakerTranslate::getWorkflows called');
      $response = $this->straker_translateClient->get('/workflow');
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting workflows for community %message.', ['%message' => $e->getMessage()]);
      throw new StrakerTranslateApiException('Failed to get workflows: ' . $e->getMessage());
    }
    $this->logger->debug('getWorkflows response received, code %code and body %body', ['%code' => $response->getStatusCode(), '%body' => (string) $response->getBody()]);
    $formatted_response = [];
    $json_response = json_decode($response->getBody(), TRUE);
    if (!empty($json_response['workflows'])) {
      foreach ($json_response['workflows'] as $worflow) {
        if (!empty($worflow['id']) && !empty($worflow['name'])) {
          $formatted_response[$worflow['id']] = $worflow['name'];
        }
      }
    }
    \Drupal::cache()->set($cid, $formatted_response, time() + 3600);
    return $formatted_response;
  }

}
