<?php

namespace Drupal\straker_translate;

/**
 * Service for managing Straker Translate interface translations.
 */
interface StrakerTranslateInterfaceTranslationServiceInterface {

  /**
   * Checks the source is uploaded correctly.
   *
   * @param string $component
   *   The component which status we want to check.
   *
   * @return bool
   *   True if the component is uploaded successfully.
   */
  public function checkSourceStatus($component);

  /**
   * Gets the source status of the given component.
   *
   * @param string $component
   *   The component which status we want to check.
   *
   * @return int
   *   The status of the target translation (see Straker Translate class constants)
   */
  public function getSourceStatus($component);

  /**
   * Sets the translation status of a given component.
   *
   * @param string $component
   *   The component which status we want to change.
   * @param int $status
   *   Status of the translation. Use Straker Translate class constants.
   *
   * @return string
   */
  public function setSourceStatus($component, $status);

  /**
   * Gets the translation status of a given component translation for a locale.
   *
   * @param string $component
   *   The component which status we want to get.
   * @param string $locale
   *   Straker translation language which we want to get.
   *
   * @return int
   *   The status of the target translation (see Straker Translate class constants)
   */
  public function getTargetStatus($component, $locale);

  /**
   * Gets the translation statuses of a given component translation for all locales.
   *
   * @param string $component
   *   The component which status we want to get.
   *
   * @return array
   *   The statuses of the target translation keyed by langcode
   *   (see Straker Translate class constants for the values)
   */
  public function getTargetStatuses($component);

  /**
   * Sets the translation status of a given component translation for a locale.
   *
   * @param string $component
   *   The component which status we want to change.
   * @param string $langcode
   *   Language code which we want to modify.
   * @param int $status
   *   Status of the translation. Use Straker Translate constants.
   *
   * @return string
   *   The component
   */
  public function setTargetStatus($component, $langcode, $status);

  /**
   * Sets the translation status of all translations of a given component.
   *
   * @param string $component
   *   The component which status we want to change.
   * @param int $status
   *   Status of the translation. Use Straker Translate constants.
   *
   * @return string
   *   The component
   */
  public function setTargetStatuses($component, $status);

  /**
   * Marks the translation status as dirty if they exist.
   *
   * @param string $component
   *   The component which status we want to change.
   *
   * @return void
   */
  public function markTranslationsAsDirty($component);

  /**
   * Gets the document id in the Straker Translate platform for a given component.
   *
   * @param string $component
   *   The component which we want the document id.
   *
   * @return string
   *   The document id in the Straker Translate platform.
   */
  public function getDocumentId($component);

  /**
   * Sets the Straker Translate document id for a given component.
   *
   * @param string $component
   *   The component which we want to set a document id.
   * @param $doc_id
   *   The document id in the Straker Translate platform.
   *
   * @return string
   *   The component.
   */
  public function setDocumentId($component, $doc_id);

  /**
   * Gets the translation source locale of a given component.
   *
   * @param string $component
   *   The component which we want to get the source locale.
   *
   * @return string
   *   The locale as expected by the Straker Translate service.
   */
  public function getSourceLocale($component);

  /**
   * Returns the source data that will be uploaded to the Straker Translate service.
   *
   * Only those fields that have actual translatable text, and have marked for upload will
   * be included.
   *
   * @param string $component
   *   The component which we want the source data.
   *
   * @return array
   */
  public function getSourceData($component);

  /**
   * Updates the component hash.
   *
   * @param string $component
   *   The component being checked.
   */
  public function updateEntityHash($component);

  /**
   * Checks if the source component data has changed from last time we uploaded it.
   *
   * @param string $component
   *   The component being checked.
   *
   * @return bool
   *   TRUE if the component has changed, false if not.
   */
  public function hasEntityChanged($component);

  /**
   * Downloads a document from the Straker Translate service for a given locale.
   *
   * @param string $component
   *   The component being downloaded.
   * @param string $locale
   *   Straker translation language which we want to download.
   *
   * @return bool
   *   TRUE if the document was downloaded successfully, FALSE if not.
   */
  public function downloadDocument($component, $locale);

  /**
   * Deletes all local metadata related to an component.
   *
   * @param string $component
   *   The component which we want to forget about.
   *
   * @return string
   *   The component.
   */
  public function deleteMetadata($component);

  /**
   * {@inheritdoc}
   */
  public function deleteAllMetadata();

  /**
   * Loads the component with the given document id.
   *
   * @param string $document_id
   *   The document id.
   *
   * @return string
   *   The component with the given document id.
   */
  public function loadByDocumentId($document_id);

  /**
   * Gets all local document ids.
   *
   * @return string[]
   *   Gets all local document ids.
   */
  public function getAllLocalDocumentIds();

  /**
   * Save the component translation.
   *
   * @param string $component
   *   The component we want to save a translation for.
   * @param $locale
   *   The locale of the translation being saved.
   * @param $data
   *   The data being saved.
   *
   * @return string
   *   Returns the component which translations are saved.
   */
  public function saveTargetData($component, $locale, $data);

  /**
   * Updates the 'initial upload' time metadata to the current request time.
   *
   * @param string $component
   *   The component for which we want to update the timestamp.
   * @param int $timestamp
   *   The timestamp we want to store.
   *
   * @return string
   *   Returns the component which translations are saved.
   */
  public function setLastUploaded($component, int $timestamp);

  /**
   * Updates the 'updated date' time metadata to the current request time.
   *
   * @param string $component
   *   The component for which we want to update the timestamp.
   * @param int $timestamp
   *   The timestamp we want to store.
   *
   * @return string
   *   Returns the component which translations are saved.
   */
  public function setLastUpdated($component, int $timestamp);

  /**
   * Returns the 'initial upload' time metadata.
   *
   * @param string $component
   *   The component for which we want to get the timestamp.
   *
   * @return int
   *   Returns the unix timestamp.
   */
  public function getLastUploaded($component);

  /**
   * Returns the 'updated date' time metadata.
   *
   * @param string $component
   *   The component for which we want to get the timestamp.
   *
   * @return int
   *   Returns the unix timestamp.
   */
  public function getUpdatedTime($component);

}
