<?php

namespace Drupal\straker_translate\Cli\Commands\Drush9;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\straker_translate\Cli\StrakerTranslateCliService;
use Drush\Commands\DrushCommands;
use Drush\Utils\StringUtils;

/**
 * A Drush9 compatible commandfile with Straker Translate operations.
 */
class StrakerTranslateCommands extends DrushCommands {

  /**
   * The Straker Translate CLI service.
   *
   * @var \Drupal\straker_translate\Cli\StrakerTranslateCliService
   */
  protected $cliService;

  /**
   * Drush8CommandBase constructor.
   *
   * @param \Drupal\straker_translate\Cli\StrakerTranslateCliService $cli_service
   *   The Straker Translate CLI service.
   */
  public function __construct(StrakerTranslateCliService $cli_service) {
    $this->cliService = $cli_service;
  }

  /**
   * Upload content to Straker Translate.
   *
   * @param $entity_type_id
   *   The entity type ID. E.g. "node"
   * @param $entity_id
   *   The entity ID. E.g. "2
   *
   * @usage drush ltk-upload node 1
   *   Upload node with ID 1.
   * @usage drush ltk-upload taxonomy_term 3
   *   Upload taxonomy term with ID 3
   *
   * @command straker_translate:upload
   * @aliases ltk-upload,straker_translate-upload
   */
  public function upload($entity_type_id, $entity_id) {
    $this->cliService->setupOutput($this->output());
    $this->cliService->setLogger($this->logger());

    $this->cliService->upload($entity_type_id, $entity_id);
  }

  /**
   * Check upload status to Straker Translate.
   *
   * @param $entity_type_id
   *   The entity type ID. E.g. "node"
   * @param $entity_id
   *   The entity ID. E.g. "2"
   *
   * @usage drush ltk-check-upload node 1
   *   Check upload status for node with ID 1.
   * @usage drush ltk-check-upload taxonomy_term 3
   *   Check upload status for taxonomy term with ID 3.
   *
   * @command straker_translate:check-upload
   * @aliases ltk-source,straker_translate-check-upload
   */
  public function checkUpload($entity_type_id, $entity_id) {
    $this->cliService->setupOutput($this->output());
    $this->cliService->setLogger($this->logger());

    $this->cliService->checkUpload($entity_type_id, $entity_id);
  }

  /**
   * Download translations from Straker Translate.
   *
   * @param $entity_type_id
   *   The entity type ID. E.g. "node"
   * @param $entity_id
   *   The entity ID. E.g. "2"
   *
   * @usage drush ltk-download node 1
   *   Download translations for node with ID 1.
   * @usage drush ltk-download taxonomy_term 3 --langcodes=es,it
   *   Download Spanish and Italian translations for taxonomy term with ID 3.
   *
   * @command straker_translate:download-translations
   * @aliases ltk-download,straker_translate-download-translations
   */
  public function downloadTranslations($entity_type_id, $entity_id, $options = ['langcodes' => 'all']) {
    $this->cliService->setupOutput($this->output());
    $this->cliService->setLogger($this->logger());

    $langcodes = StringUtils::csvToArray($options['langcodes']);
    $this->cliService->downloadTranslations($entity_type_id, $entity_id, $langcodes);
  }

}
