<?php

namespace Drupal\straker_translate\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\straker_translate\LanguageLocaleMapperInterface;
use Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface;
use Drupal\straker_translate\StrakerTranslateInterface;
use Drupal\straker_translate\StrakerTranslateLocale;
use Drupal\Core\Language\LanguageInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Session\AccountInterface;

/**
 * Returns responses for straker_translate module setup routes.
 */
class StrakerTranslateDashboardController extends StrakerTranslateControllerBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * @var \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface
   */
  protected $straker_translate_configuration;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * Constructs a StrakerTranslateDashboardController object.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The Request instance.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\straker_translate\StrakerTranslateInterface $straker_translate
   *   The straker_translate service.
   * @param \Drupal\straker_translate\LanguageLocaleMapperInterface $language_locale_mapper
   *   The language-locale mapper.
   * @param \Drupal\straker_translate\StrakerTranslateConfigurationServiceInterface $straker_translate_configuration
   *   The Straker Translate configuration service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $url_generator
   *   The url generator.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(Request $request, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, StrakerTranslateInterface $straker_translate, LanguageLocaleMapperInterface $language_locale_mapper, StrakerTranslateConfigurationServiceInterface $straker_translate_configuration, FormBuilderInterface $form_builder, LoggerInterface $logger, UrlGeneratorInterface $url_generator, AccountInterface $current_user) {
    parent::__construct($request, $config_factory, $straker_translate, $language_locale_mapper, $form_builder, $logger);
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->straker_translate_configuration = $straker_translate_configuration;
    $this->urlGenerator = $url_generator;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('straker_translate'),
      $container->get('straker_translate.language_locale_mapper'),
      $container->get('straker_translate.configuration'),
      $container->get('form_builder'),
      $container->get('logger.channel.straker_translate'),
      $container->get('url_generator'),
      $container->get('current_user')
    );
  }

  /**
   * Presents a dashboard overview page of translation status through Straker Translate.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The page request.
   *
   * @return array
   *   The dashboard form, or a redirect to the connect page.
   */
  public function dashboardPage(Request $request) {
    if ($redirect = $this->checkSetup()) {
      return $redirect;
    }
    $cms_data = $this->getDashboardInfo();
    $build = [];
    $build['#attached']['library'][] = 'straker_translate/straker_translate.dashboard';
    $build['#attached']['drupalSettings']['straker_translate']['cms_data'] = $cms_data;
    $build['#title'] = $this->t('Dashboard');
    $build['ltk-dashboard'] = [
      '#type' => 'container',
      '#attributes' => [
        'ltk-dashboard' => '',
        'ng-app' => 'StrakerTranslateApp',
        'style' => 'margin-top: -15px;',
      ],
    ];
    return $build;
  }

  /**
   *
   */
  public function endpoint(Request $request) {
    if ($redirect = $this->checkSetup()) {
      return $redirect;
    }
    $request_method = $request->getMethod();
    $language_permission = $this->currentUser->hasPermission('administer languages');

    $http_status_code = Response::HTTP_NOT_IMPLEMENTED;
    $response = [
      'method' => $request_method,
    ];
    switch ($request_method) {
      case 'POST':
        if ($language_permission) {
          $languageStorage = $this->entityTypeManager->getStorage('configurable_language');
          $straker_translate_locale = $request->get('code');
          $native = $request->get('native');
          $language = $request->get('language');
          $direction = $request->get('direction');
          if (isset($language, $straker_translate_locale, $direction)) {
            // First, we try if there is a disabled language with that locale.
            $existingLanguage = $languageStorage->getQuery()
              ->accessCheck(FALSE)
              ->condition('third_party_settings.straker_translate.disabled', TRUE)
              ->condition('third_party_settings.straker_translate.locale', $straker_translate_locale)
              ->execute();
            if (!$existingLanguage) {
              // If we didn't find it, maybe the language was the default
              // locale, and it didn't have a locale stored.
              $existingLanguage = $languageStorage->getQuery()
                ->accessCheck(FALSE)
                ->condition('third_party_settings.straker_translate.disabled', TRUE)
                ->condition('id', StrakerTranslateLocale::convertStrakerTranslate2Drupal($straker_translate_locale, FALSE))
                ->execute();
            }
            if ($existingLanguage) {
              $language = $languageStorage->load(reset($existingLanguage));
            }
            else {
              $rtl = ($direction == 'RTL') ? LanguageInterface::DIRECTION_RTL : LanguageInterface::DIRECTION_LTR;
              $langcode = StrakerTranslateLocale::generateStrakerTranslate2Drupal($straker_translate_locale);
              $language = $languageStorage->create([
                'id' => $langcode,
                'label' => $language,
                'native' => $native,
                'direction' => $rtl,
              ]);
            }
            $language->setThirdPartySetting('straker_translate', 'disabled', FALSE);
            $language->setThirdPartySetting('straker_translate', 'locale', $straker_translate_locale);
            $language->save();
            $response += $this->getLanguageReport($language);
            $http_status_code = Response::HTTP_OK;
          }
          // @todo (1) add language to CMS if not enabled X, (2) add language to Verify project
        }
        else {
          $response['message'] = "Administer Languages permission required to add language";
          $http_status_code = Response::HTTP_FORBIDDEN;
        }
        break;

      case 'DELETE':
        if ($language_permission) {
          $content = $request->getContent();
          $parsed_content = [];
          parse_str($content, $parsed_content);
          $locale = $parsed_content['code'];
          $language = $this->languageLocaleMapper->getConfigurableLanguageForLocale($locale);
          $response['language'] = $language->id();
          $this->straker_translate_configuration->disableLanguage($language);
          $this->languageManager()->reset();
          $response['message'] = "Language disabled: $locale";
          $http_status_code = Response::HTTP_OK;
        }
        else {
          $response['message'] = "Administer Languages permission required to delete language";
        }
        break;

      case 'GET':
      default:
        // isset($request->get('code')) ? $_REQUEST['code'] : NULL;.
        $locale_code = $request->get('code');
        $details = $this->getLanguageDetails($locale_code);
        if (empty($details)) {
          $response['error'] = "language code not found.";
          return (new JsonResponse($response, Response::HTTP_NOT_FOUND));
        }
        $http_status_code = Response::HTTP_OK;
        $response = array_merge($response, $details);
        break;
    }

    return (new JsonResponse($response, $http_status_code));
  }

  /**
   *
   */
  private function getLanguageDetails($straker_translate_locale_requested = NULL) {
    $response = [];
    $available_languages = $this->languageManager->getLanguages();
    $source_total = 0;
    $target_total = 0;
    $source_totals = [];
    $target_totals = [];

    // If we get a parameter, only return that language. Otherwise return all languages.
    foreach ($available_languages as $language) {
      // We check if we have a saved straker_translate locale.
      // If not, we default to the id conversion.
      // Language manager returns Language objects, not ConfigurableLanguage,
      // because the language manager is initiated before the config system, and
      // loads the configuration bypassing it.
      $straker_translate_locale = $this->languageLocaleMapper->getLocaleForLangcode($language->getId());

      if (!is_null($straker_translate_locale_requested) && $straker_translate_locale_requested != $straker_translate_locale) {
        continue;
      }

      $language_report = $this->getLanguageReport($language);
      if ($straker_translate_locale_requested === $straker_translate_locale) {
        $response = $language_report;
      }
      else {
        if (!empty($straker_translate_locale)) {
          $response[$straker_translate_locale] = $language_report;
        }
        else {
          // There are some edge cases where there is no locale.
          // We default to the only known code, which is only the langcode.
          $response[$language->getId()] = $language_report;
        }
      }
      $source_total += $language_report['source']['total'];
      $target_total += $language_report['target']['total'];
      $source_totals = self::calcLanguageTotals($source_totals, $language_report['source']['types']);
      $target_totals = self::calcLanguageTotals($target_totals, $language_report['target']['types']);
    }
    if (is_null($straker_translate_locale_requested)) {
      $response = [
        'languages' => $response,
        'source' => ['types' => $source_totals, 'total' => $source_total],
        'target' => ['types' => $target_totals, 'total' => $target_total],
        'count' => count($available_languages),
      ];
    }
    return $response;
  }

  /**
   *
   */
  protected function getDashboardInfo() {
    global $base_url, $base_root;
    $accountConfig = $this->configFactory->get('straker_translate.account');
    return [
      "community_id" => $accountConfig->get('default.community'),
      "external_id" => $accountConfig->get('login_id'),
      "vault_id" => $accountConfig->get('default.vault'),
      "workflow_id" => $accountConfig->get('default.workflow'),
      "project_id" => $accountConfig->get('default.project'),
      "first_name" => 'Drupal User',
      "last_name" => '',
      "email" => $accountConfig->get('login_id'),
      // CMS data that will be used for building the dashboard with JS.
      "cms_site_id" => $base_url,
      "cms_site_key" => $base_url,
      "cms_site_name" => 'Drupal Site',
      "cms_type" => 'Drupal',
      "cms_version" => 'VERSION HERE',
      "cms_tag" => 'CMS TAG HERE',
      // FIX: should be the currently selected locale.
      "locale" => "en_US",
      "module_version" => '1.x',
      "endpoint_url" => $this->urlGenerator->generateFromRoute('straker_translate.dashboard_endpoint'),
    ];
  }

  /**
   *
   */
  protected function getLanguageReport(LanguageInterface $language, $active = 1, $enabled = 1) {
    $langcode = $language->getId();
    $locale = $this->languageLocaleMapper->getLocaleForLangcode($langcode);
    $configLanguage = $this->entityTypeManager->getStorage('configurable_language')->load($langcode);
    $types = $this->getEnabledTypes();

    $stat = [
      'locale' => $locale,
      'xcode' => $langcode,
      'active' => $this->straker_translate_configuration->isLanguageEnabled($configLanguage) ? 1 : 0,
      'enabled' => 1,
      'source' => [
        'types' => $this->getSourceTypeCounts($langcode),
        'total' => 0,
      ],
      'target' => [
        'types' => $this->getTargetTypeCounts($langcode),
        'total' => 0,
      ],
    ];
    foreach ($types as $type) {
      $stat['source']['total'] += $stat['source']['types'][$type] ?? 0;
      $stat['target']['total'] += $stat['target']['types'][$type] ?? 0;
    }
    return $stat;
  }

  /**
   * Gets the entity type ids of entities to be translated with Straker Translate.
   *
   * @return array The entity type names of content entities enabled.
   */
  protected function getEnabledTypes() {
    $types = $this->straker_translate_configuration->getEnabledEntityTypes();
    return empty($types) ? $types : array_keys($types);
  }

  /**
   *
   */
  protected function getSourceTypeCounts($langcode, $types = NULL) {
    $types = is_null($types) ? $this->getEnabledTypes() : $types;
    $result = [];
    foreach ($types as $type) {
      $result[$type] = $this->getSourceTypeCount($langcode, $type);
    }
    return $result;
  }

  /**
   *
   */
  protected function getSourceTypeCount($langcode, $type) {
    $query = $this->entityTypeManager->getStorage($type)->getQuery();
    $query->accessCheck(FALSE);
    $count = $query->condition('langcode', $langcode)
      ->condition('default_langcode', 1)
      ->count()
      ->execute();
    return (int) $count;
  }

  /**
   *
   */
  protected function getTargetTypeCounts($langcode, $types = NULL) {
    $types = is_null($types) ? $this->getEnabledTypes() : $types;
    $result = [];
    foreach ($types as $type) {
      $result[$type] = $this->getTargetTypeCount($langcode, $type);
    }
    return $result;
  }

  /**
   *
   */
  protected function getTargetTypeCount($langcode, $type) {
    $query = $this->entityTypeManager->getStorage($type)->getQuery();
    $query->accessCheck(FALSE);
    $count = $query->condition('langcode', $langcode)
      ->condition('default_langcode', 0)
      ->count()
      ->execute();
    return (int) $count;
  }

  /**
   * Sums the values of the arrays be there keys (PHP 4, PHP 5)
   * array array_sum_values ( array array1 [, array array2 [, array ...]] )
   */
  private static function calcLanguageTotals() {
    $return = [];
    $intArgs = func_num_args();
    $arrArgs = func_get_args();
    if ($intArgs < 1) {
      trigger_error('Warning: Wrong parameter count for calcLanguageTotals()', E_USER_WARNING);
    }

    foreach ($arrArgs as $arrItem) {
      if (!is_array($arrItem)) {
        trigger_error('Warning: Wrong parameter values for calcLanguageTotals()', E_USER_WARNING);
      }
      foreach ($arrItem as $k => $v) {
        if (!array_key_exists($k, $return)) {
          $return[$k] = 0;
        }
        $return[$k] += $v;
      }
    }
    return $return;
  }

}
